/*
  Decoder

  AC3 decoder class. Manage Parser, Mixer and IMDCT objects.

  Usage.

  is_pes       - decode MPEG2 PES stream instead of pure ac3 stream.
  dynrng       - use dynamic range conpression (DRC)
  dynrng_power - power of dynamic range compression (dynrng = dynrng^dynrng_power [here 'dynrng' - coeffitient from parser])
  clev_lock    - mixer uses clev from parser (only when auto_matrix = true)
  slev_lock    - mixer uses slev from parser (only when auto_matrix = true)
  lfelev_lock  - mixer uses lfelev = 1.0 (only when auto_matrix = true)

  auto_matrix  - use standart mixing matrix recalculated each frame
  normalize_matrix - normalize mixing matrix for no overflow at output (only when auto_matrix = true)

  frames       - number of decoded frames (total, reset() does not reset it)
  errors       - number of erorrs (total, reset() does not reset it)

  samples      - when block() function called it is 256 decoded samples buffer
  delay        - delayed samples (do not modify it at decode()!)

  reset()      - reset decoder, prepare to decode other stream
  decode()     - decode stream
  block()      - pure; called each time block decoded
*/


#ifndef DECODER_H
#define DECODER_H

#include "defs.h"
#include "parser.h"
#include "mixer.h"
#include "imdct.h"
#include "demux.h"


const acmod_t         default_mode  = MODE_STEREO;
const channel_order_t default_order = {CH_L, CH_R, 0, 0, 0, 0};



class Decoder
{
protected:
  // frame buffer
  uint8_t  buf[3840];

  // sample buffers
  sample_buffer_t samples;
  sample_buffer_t delay;

  Demux    demux;
  Parser   parser;
  Mixer    mixer;
  IMDCT    imdct;
  BSI      bsi;

public:
  bool     is_pes;            // false by default
  bool     dynrng;            // true
  sample_t dynrng_power;      // 1.0

  bool     clev_lock;         // true
  bool     slev_lock;         // true
  bool     lfelev_lock;       // true

  bool     auto_matrix;       // true
  bool     normalize_matrix;  // true

  int frames;
  int errors;

  Decoder(acmod_t speakers = default_mode, const channel_order_t order = default_order);

  const BSI *get_bsi()     { return &bsi; };

  acmod_t      get_acmod() { return mixer.get_output_mode(); };
  virtual void set_acmod(acmod_t acmod, const channel_order_t order);

  virtual void reset();
  virtual void decode(uint8_t *buffer, uint32_t length);
  virtual void frame();
  virtual void block() = 0;


private:
  // source buffers
  uint8_t *bufptr;      // current data pointer
  uint8_t *bufpos;      // required data pointer
};


#endif
