#include <string.h>
#include "decoder.h"


Decoder::Decoder(acmod_t acmod, const channel_order_t order)
{
  frames = 0;
  errors = 0;

  is_pes            = false;

  dynrng            = true;
  dynrng_power      = 1.0;

  slev_lock         = true;
  clev_lock         = true;
  lfelev_lock       = true;

  auto_matrix       = true;
  normalize_matrix  = true;

  set_acmod(acmod, order);
}

void 
Decoder::reset()
{
  bufptr = buf;
  bufpos = buf + 7;
  memset(&bsi, 0, sizeof(bsi));
  memset(&delay, 0, sizeof(delay));
  memset(&samples, 0, sizeof(samples));

  parser.reset();
  demux.reset();
  mixer.reset();
}

void 
Decoder::set_acmod(acmod_t acmod, const channel_order_t order)
{
  mixer.set_output_mode(acmod, order);
  reset();
}

void 
Decoder::decode(uint8_t *buffer, uint32_t length)
{
  if (is_pes)
    length = demux.decode(buffer, length);

  long len;

  // while we have something in buffer we will try to decode it
  while (1) 
  {
    // copy requested data to buf from buffer if we have requested amount
    // or copy rest of buffer to remember it for future calls so we have processed all given data

    len = length;

    if (!len) break;

    if (len > bufpos - bufptr)
      len = bufpos - bufptr;

    memcpy (bufptr, buffer, len);
    bufptr += len;
    buffer += len;
    length -= len;

    // if we have all requested data
    if (bufptr == bufpos) 
    {
      // find sync, decode frame header and load frame data
      if (bufpos == buf + 7) 
      {
        int frame_size = parser.syncinfo (buf);

        // skip if it is not a sync and cycle
        if (!frame_size) 
        {
          for (bufptr = buf; bufptr < buf + 6; bufptr++)
             bufptr[0] = bufptr[1];
          continue;
        }

        // or request more data
        bufpos = buf + frame_size;
      } 
      // decode frame data
      else 
      {
        if (!parser.frame (buf))
        {
          parser.get_bsi(bsi);
          frame();
          frames++;
        }        
        else
          errors++;
        bufptr = buf;
        bufpos = buf + 7;
      }
    }
  }
}

void 
Decoder::frame()
{
  // Set frame-level mixing params
  mixer.set_input_mode(parser.acmod);
  if (clev_lock)   mixer.clev = ((mixer.get_output_mode() & 1) && (mixer.get_output_mode() != 1))? 1.0: parser.clev;
  if (slev_lock)   mixer.slev = (mixer.get_output_mode() & 4)? 1.0: parser.slev;
  if (lfelev_lock) mixer.lfelev = 1.0;
  if (!dynrng)     mixer.dynrng = 1.0;

  if (auto_matrix)
  {
    mixer.calc_matrix();
    if (normalize_matrix)
      mixer.normalize_matrix();
  }

  for (int i = 0; i < 6; i++) 
  {
    // perfomance warinig:
    // after previous mixing all channels may 
    // be changed so clear all?
    // fixes bug 07.07.02 on Dracula film
    memset(samples, 0, sizeof(samples));
    memset(delay, 0, sizeof(delay));

    if (parser.parse_block(samples))
    {
      errors++;
      break;
    }

    parser.get_bsi(bsi);

    // IMDCT
    for (int ch = 0; ch < MODE_NFCHANS(parser.acmod); ch++) 
      if (MODE_CH(parser.acmod) && CH_MASK(ch))
        if (parser.blksw[ch])
          imdct.imdct_256(samples[ch], delay[ch]);
        else 
          imdct.imdct_512(samples[ch], delay[ch]);

    if (IS_MODE_LFE(parser.acmod))
      imdct.imdct_512(samples[MODE_NFCHANS(parser.acmod)], delay[MODE_NFCHANS(parser.acmod)]);

    // Mix and gain control with overflow check
    if (dynrng)
      mixer.dynrng = pow(parser.dynrng, dynrng_power);

    mixer.mix(samples, delay);

    // do something useful
    block();
  }
}
