#include <string.h>

#include "ac3dec.h"    
#include "tables.h"

////////////////////bit allocate///////////////////////////////

static const int hthtab[3][50] = {
    {0x730, 0x730, 0x7c0, 0x800, 0x820, 0x840, 0x850, 0x850, 0x860, 0x860,
     0x860, 0x860, 0x860, 0x870, 0x870, 0x870, 0x880, 0x880, 0x890, 0x890,
     0x8a0, 0x8a0, 0x8b0, 0x8b0, 0x8c0, 0x8c0, 0x8d0, 0x8e0, 0x8f0, 0x900,
     0x910, 0x910, 0x910, 0x910, 0x900, 0x8f0, 0x8c0, 0x870, 0x820, 0x7e0,
     0x7a0, 0x770, 0x760, 0x7a0, 0x7c0, 0x7c0, 0x6e0, 0x400, 0x3c0, 0x3c0},
    {0x710, 0x710, 0x7a0, 0x7f0, 0x820, 0x830, 0x840, 0x850, 0x850, 0x860,
     0x860, 0x860, 0x860, 0x860, 0x870, 0x870, 0x870, 0x880, 0x880, 0x880,
     0x890, 0x890, 0x8a0, 0x8a0, 0x8b0, 0x8b0, 0x8c0, 0x8c0, 0x8e0, 0x8f0,
     0x900, 0x910, 0x910, 0x910, 0x910, 0x900, 0x8e0, 0x8b0, 0x870, 0x820,
     0x7e0, 0x7b0, 0x760, 0x770, 0x7a0, 0x7c0, 0x780, 0x5d0, 0x3c0, 0x3c0},
    {0x680, 0x680, 0x750, 0x7b0, 0x7e0, 0x810, 0x820, 0x830, 0x840, 0x850,
     0x850, 0x850, 0x860, 0x860, 0x860, 0x860, 0x860, 0x860, 0x860, 0x860,
     0x870, 0x870, 0x870, 0x870, 0x880, 0x880, 0x880, 0x890, 0x8a0, 0x8b0,
     0x8c0, 0x8d0, 0x8e0, 0x8f0, 0x900, 0x910, 0x910, 0x910, 0x900, 0x8f0,
     0x8d0, 0x8b0, 0x840, 0x7f0, 0x790, 0x760, 0x7a0, 0x7c0, 0x7b0, 0x720}
};

static const int8_t baptab[305] = {
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16,	/* 93 padding elems */

    16, 16, 16, 16, 16, 16, 16, 16, 16, 14, 14, 14, 14, 14, 14, 14,
    14, 12, 12, 12, 12, 11, 11, 11, 11, 10, 10, 10, 10,  9,  9,  9,
     9,  8,  8,  8,  8,  7,  7,  7,  7,  6,  6,  6,  6,  5,  5,  5,
     5,  4,  4, -3, -3,  3,  3,  3, -2, -2, -1, -1, -1, -1, -1,  0,

     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
     0,  0,  0,  0					/* 148 padding elems */
};

static const int bndtab[30] = {21, 22,  23,  24,  25,  26,  27,  28,  31,  34,
			 37, 40,  43,  46,  49,  55,  61,  67,  73,  79,
			 85, 97, 109, 121, 133, 157, 181, 205, 229, 253};

static const int8_t latab[256] = {
    -64, -63, -62, -61, -60, -59, -58, -57, -56, -55, -54, -53,
    -52, -52, -51, -50, -49, -48, -47, -47, -46, -45, -44, -44,
    -43, -42, -41, -41, -40, -39, -38, -38, -37, -36, -36, -35,
    -35, -34, -33, -33, -32, -32, -31, -30, -30, -29, -29, -28,
    -28, -27, -27, -26, -26, -25, -25, -24, -24, -23, -23, -22,
    -22, -21, -21, -21, -20, -20, -19, -19, -19, -18, -18, -18,
    -17, -17, -17, -16, -16, -16, -15, -15, -15, -14, -14, -14,
    -13, -13, -13, -13, -12, -12, -12, -12, -11, -11, -11, -11,
    -10, -10, -10, -10, -10,  -9,  -9,  -9,  -9,  -9,  -8,  -8,
     -8,  -8,  -8,  -8,  -7,  -7,  -7,  -7,  -7,  -7,  -6,  -6,
     -6,  -6,  -6,  -6,  -6,  -6,  -5,  -5,  -5,  -5,  -5,  -5,
     -5,  -5,  -4,  -4,  -4,  -4,  -4,  -4,  -4,  -4,  -4,  -4,
     -4,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,  -3,
     -3,  -3,  -3,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,
     -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -2,  -1,  -1,
     -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
     -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
     -1,  -1,  -1,  -1,  -1,  -1,   0,   0,   0,   0,   0,   0,
      0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
      0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
      0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
      0,   0,   0,   0
};

#define UPDATE_LEAK() 		\
do {				\
    fastleak += fdecay;		\
    if (fastleak > psd + fgain)	\
	fastleak = psd + fgain;	\
    slowleak += sdecay;		\
    if (slowleak > psd + sgain)	\
	slowleak = psd + sgain;	\
} while (0)

#define COMPUTE_MASK()				\
do {						\
    if (psd > dbknee)				\
	mask -= (psd - dbknee) >> 2;		\
    if (mask > hth [i >> halfrate])		\
	mask = hth [i >> halfrate];		\
    mask -= snroffset + 128 * deltba[i];	\
    mask = (mask > 0) ? 0 : ((-mask) >> 5);	\
    mask -= floor;				\
} while (0)

void a52_bit_allocate (a52_state_t * state, ba_t * ba, int bndstart,
		       int start, int end, int fastleak, int slowleak,
		       expbap_t * expbap)
{
    static const int slowgain[4] = {0x540, 0x4d8, 0x478, 0x410};
    static const int dbpbtab[4]  = {0xc00, 0x500, 0x300, 0x100};
    static const int floortab[8] = {0x910, 0x950, 0x990, 0x9d0,
                                    0xa10, 0xa90, 0xb10, 0x1400};

    int i, j;
    uint8_t * exp;
    int8_t * bap;
    int fdecay, fgain, sdecay, sgain, dbknee, floor, snroffset;
    int psd, mask;
    const int8_t * deltba;
    const int * hth;
    int halfrate;

    halfrate = state->halfrate;
    fdecay = (63 + 20 * ((state->bai >> 7) & 3)) >> halfrate;	/* fdcycod */
    fgain = 128 + 128 * (ba->bai & 7);				/* fgaincod */
    sdecay = (15 + 2 * (state->bai >> 9)) >> halfrate;		/* sdcycod */
    sgain = slowgain[(state->bai >> 5) & 3];			/* sgaincod */
    dbknee = dbpbtab[(state->bai >> 3) & 3];			/* dbpbcod */
    hth = hthtab[state->fscod];
    /*
     * if there is no delta bit allocation, make deltba point to an area
     * known to contain zeroes. baptab+156 here.
     */
    deltba = (ba->deltbae == DELTA_BIT_NONE) ? baptab + 156 : ba->deltba;
    floor = floortab[state->bai & 7];				/* floorcod */
    snroffset = 960 - 64 * state->csnroffst - 4 * (ba->bai >> 3) + floor;
    floor >>= 5;

    exp = expbap->exp;
    bap = expbap->bap;

    i = bndstart;
    j = start;
    if (start == 0) {	/* not the coupling channel */
	int lowcomp;

	lowcomp = 0;
	j = end - 1;
	do {
	    if (i < j) {
		if (exp[i+1] == exp[i] - 2)
		    lowcomp = 384;
		else if (lowcomp && (exp[i+1] > exp[i]))
		    lowcomp -= 64;
	    }
	    psd = 128 * exp[i];
	    mask = psd + fgain + lowcomp;
	    COMPUTE_MASK ();
	    bap[i] = (baptab+156)[mask + 4 * exp[i]];
	    i++;
	} while ((i < 3) || ((i < 7) && (exp[i] > exp[i-1])));
	fastleak = psd + fgain;
	slowleak = psd + sgain;

	while (i < 7) {
	    if (i < j) {
		if (exp[i+1] == exp[i] - 2)
		    lowcomp = 384;
		else if (lowcomp && (exp[i+1] > exp[i]))
		    lowcomp -= 64;
	    }
	    psd = 128 * exp[i];
	    UPDATE_LEAK ();
	    mask = ((fastleak + lowcomp < slowleak) ?
		    fastleak + lowcomp : slowleak);
	    COMPUTE_MASK ();
	    bap[i] = (baptab+156)[mask + 4 * exp[i]];
	    i++;
	}

	if (end == 7)	/* lfe channel */
	    return;

	do {
	    if (exp[i+1] == exp[i] - 2)
		lowcomp = 320;
	    else if (lowcomp && (exp[i+1] > exp[i]))
		lowcomp -= 64;
	    psd = 128 * exp[i];
	    UPDATE_LEAK ();
	    mask = ((fastleak + lowcomp < slowleak) ?
		    fastleak + lowcomp : slowleak);
	    COMPUTE_MASK ();
	    bap[i] = (baptab+156)[mask + 4 * exp[i]];
	    i++;
	} while (i < 20);

	while (lowcomp > 128) {		/* two iterations maximum */
	    lowcomp -= 128;
	    psd = 128 * exp[i];
	    UPDATE_LEAK ();
	    mask = ((fastleak + lowcomp < slowleak) ?
		    fastleak + lowcomp : slowleak);
	    COMPUTE_MASK ();
	    bap[i] = (baptab+156)[mask + 4 * exp[i]];
	    i++;
	}
	j = i;
    }

    do {
	int startband, endband;

	startband = j;
	endband = ((bndtab-20)[i] < end) ? (bndtab-20)[i] : end;
	psd = 128 * exp[j++];
	while (j < endband) {
	    int next, delta;

	    next = 128 * exp[j++];
	    delta = next - psd;
	    switch (delta >> 9) {
	    case -6: case -5: case -4: case -3: case -2:
		psd = next;
		break;
	    case -1:
		psd = next + latab[(-delta) >> 1];
		break;
	    case 0:
		psd += latab[delta >> 1];
		break;
	    }
	}
	/* minpsd = -289 */
	UPDATE_LEAK ();
	mask = (fastleak < slowleak) ? fastleak : slowleak;
	COMPUTE_MASK ();
	i++;
	j = startband;
	do {
	    /* max(mask+4*exp)=147=-(minpsd+fgain-deltba-snroffset)>>5+4*exp */
	    /* min(mask+4*exp)=-56=-(sgain-deltba-snroffset)>>5 */
	    bap[j] = (baptab+156)[mask + 4 * exp[j]];
	} while (++j < endband);
    } while (j < end);
}
////////////////////////parser///////////////////////////////

int 
CAC3Decoder::parse_exponents (int expstr, int ngrps, uint8_t exponent, uint8_t * dest)
{
  int exps;

  while (ngrps--) 
  {
    exps = bitstream.get (7);

    exponent += exp_1[exps];
    if (exponent > 24)
      return 1;

    switch (expstr) 
    {
      case EXP_D45:
        *(dest++) = exponent;
        *(dest++) = exponent;
      case EXP_D25:
        *(dest++) = exponent;
      case EXP_D15:
        *(dest++) = exponent;
    }

    exponent += exp_2[exps];
    if (exponent > 24)
      return 1;

    switch (expstr) 
    {
      case EXP_D45:
        *(dest++) = exponent;
        *(dest++) = exponent;
      case EXP_D25:
        *(dest++) = exponent;
      case EXP_D15:
        *(dest++) = exponent;
    }

    exponent += exp_3[exps];
    if (exponent > 24)
      return 1;

    switch (expstr) 
    {
      case EXP_D45:
        *(dest++) = exponent;
        *(dest++) = exponent;
      case EXP_D25:
        *(dest++) = exponent;
      case EXP_D15:
        *(dest++) = exponent;
    }
  }	
  return 0;
}

int 
CAC3Decoder::parse_deltba (int8_t * deltba)
{
  int deltnseg, deltlen, delta, j;

  memset (deltba, 0, 50);

  deltnseg = bitstream.get (3);
  j = 0;
  do 
  {
    j += bitstream.get (5);
    deltlen = bitstream.get (4);
    delta = bitstream.get (3);
    delta -= (delta >= 4) ? 3 : 4;
    if (!deltlen)
      continue;
    if (j + deltlen >= 50)
      return 1;
    while (deltlen--)
      deltba[j++] = delta;
  } while (deltnseg--);

  return 0;
}

inline int 
CAC3Decoder::zero_snr_offsets (int nfchans, a52_state_t * state)
{
  int i;

  if ((state->csnroffst) ||
      (state->chincpl && state->cplba.bai >> 3) ||  /* cplinu, fsnroffst */
      (lfeon && state->lfeba.bai >> 3))  /* fsnroffst */
    return 0;

  for (i = 0; i < nfchans; i++)
    if (state->ba[i].bai >> 3)  /* fsnroffst */
      return 0;
  return 1;
}

inline int16_t 
CAC3Decoder::dither_gen (void)
{
  static uint16_t lfsr_state = 1;
  int16_t state;

  state = dither_lut[lfsr_state >> 8] ^ (lfsr_state << 8);
	
  lfsr_state = (uint16_t) state;
  return state;
}

void 
CAC3Decoder::coeff_get (sample_t * coeff, expbap_t * expbap,
                        quantizer_t * quantizer, int dither, int end)
{
  int i;
  uint8_t * exp;
  int8_t * bap;
  sample_t factor[25];

  for (i = 0; i <= 24; i++)
    factor[i] = scale_factor[i];// * level;

  exp = expbap->exp;
  bap = expbap->bap;

  for (i = 0; i < end; i++) 
  {
    int bapi;

    bapi = bap[i];
    switch (bapi) 
    {
      case 0:
        if (dither) 
        {
          coeff[i] = dither_gen() * LEVEL_3DB * factor[exp[i]];
          continue;
        } 
        else 
        {
          coeff[i] = 0;
          continue;
        }

      case -1:
        if (quantizer->q1_ptr >= 0) 
        {
          coeff[i] = quantizer->q1[quantizer->q1_ptr--] * factor[exp[i]];
          continue;
        } 
        else 
        {
          int code;

          code = bitstream.get (5);

          quantizer->q1_ptr = 1;
          quantizer->q1[0] = q_1_2[code];
          quantizer->q1[1] = q_1_1[code];
          coeff[i] = q_1_0[code] * factor[exp[i]];
          continue;
        }

      case -2:
        if (quantizer->q2_ptr >= 0) 
        {
          coeff[i] = quantizer->q2[quantizer->q2_ptr--] * factor[exp[i]];
          continue;
        } 
        else 
        {
          int code;

          code = bitstream.get (7);

          quantizer->q2_ptr = 1;
          quantizer->q2[0] = q_2_2[code];
          quantizer->q2[1] = q_2_1[code];
          coeff[i] = q_2_0[code] * factor[exp[i]];
          continue;
        }

      case 3:
        coeff[i] = q_3[bitstream.get (3)] * factor[exp[i]];
        continue;

      case -3:
        if (quantizer->q4_ptr == 0) 
        {
          quantizer->q4_ptr = -1;
          coeff[i] = quantizer->q4 * factor[exp[i]];
          continue;
        } 
        else 
        {
          int code;

          code = bitstream.get (7);

          quantizer->q4_ptr = 0;
          quantizer->q4 = q_4_1[code];
          coeff[i] = q_4_0[code] * factor[exp[i]];
          continue;
        }

      case 4:
        coeff[i] = q_5[bitstream.get (4)] * factor[exp[i]];
        continue;

      default:
        coeff[i] = ((bitstream.get_2 (bapi) << (16 - bapi)) *
        factor[exp[i]]);
    }
  }
}

void 
CAC3Decoder::coeff_get_coupling (int nfchans, quantizer_t * quantizer, uint8_t dithflag[5])
{
  int cplbndstrc, bnd, i, i_end, ch;
  uint8_t * exp;
  int8_t * bap;
  sample_t cplco[5];

  exp = state->cpl_expbap.exp;
  bap = state->cpl_expbap.bap;
  bnd = 0;
  cplbndstrc = state->cplbndstrc;
  i = state->cplstrtmant;
  while (i < state->cplendmant) 
  {
    i_end = i + 12;
    while (cplbndstrc & 1) 
    {
      cplbndstrc >>= 1;
      i_end += 12;
    }
    cplbndstrc >>= 1;
    for (ch = 0; ch < nfchans; ch++)
      cplco[ch] = state->cplco[ch][bnd];// * coeff[ch];
    bnd++;

    while (i < i_end) 
    {
      sample_t cplcoeff;
      int bapi;

      bapi = bap[i];
      switch (bapi) 
      {
        case 0:
          cplcoeff = LEVEL_3DB * scale_factor[exp[i]];
          for (ch = 0; ch < nfchans; ch++)
            if ((state->chincpl >> ch) & 1) 
            {
              if (dithflag[ch])
                samples[ch][i] = (cplcoeff * cplco[ch] * dither_gen ());
              else
                samples[ch][i] = 0;
            }
          i++;
          continue;

        case -1:
          if (quantizer->q1_ptr >= 0) 
          {
            cplcoeff = quantizer->q1[quantizer->q1_ptr--];
            break;
          } 
          else 
          {
            int code;

            code = bitstream.get (5);

            quantizer->q1_ptr = 1;
            quantizer->q1[0] = q_1_2[code];
            quantizer->q1[1] = q_1_1[code];
            cplcoeff = q_1_0[code];
            break;
          }

        case -2:
          if (quantizer->q2_ptr >= 0) 
          {
            cplcoeff = quantizer->q2[quantizer->q2_ptr--];
            break;
          } 
          else 
          {
            int code;

            code = bitstream.get (7);

            quantizer->q2_ptr = 1;
            quantizer->q2[0] = q_2_2[code];
            quantizer->q2[1] = q_2_1[code];
            cplcoeff = q_2_0[code];
            break;
          }

        case 3:
          cplcoeff = q_3[bitstream.get (3)];
          break;

        case -3:
          if (quantizer->q4_ptr == 0) 
          {
            quantizer->q4_ptr = -1;
            cplcoeff = quantizer->q4;
            break;
          } 
          else 
          {
            int code;

            code = bitstream.get (7);

            quantizer->q4_ptr = 0;
            quantizer->q4 = q_4_1[code];
            cplcoeff = q_4_0[code];
            break;
          }

        case 4:
          cplcoeff = q_5[bitstream.get (4)];
          break;

        default:
          cplcoeff = bitstream.get_2 (bapi) << (16 - bapi);
      }

      cplcoeff *= scale_factor[exp[i]];

      for (ch = 0; ch < nfchans; ch++)
        if ((state->chincpl >> ch) & 1)
          samples[ch][i] = cplcoeff * cplco[ch];
      i++;
    }
  }
}
//////////////////////////////////////////////////////////


CAC3Decoder::CAC3Decoder()
{
  state = new a52_state_t;
  if (!state) throw "CAC3Deocder::CAC3Deocder: no memory";

  int i;
  for (i = 0; i < 256 * 6; i++)
    ((double*)samples)[i] = 0;
  for (i = 0; i < 256 * 6; i++)
    ((double*)samples1)[i] = 0;

  imdct.init ();


  bufptr = buf;
  bufpos = buf + 7;

  level = 1.0;
  bias  = 0.0;
  dynrng = true;
  acmod_out = MODE_STEREO;
}

CAC3Decoder::~CAC3Decoder()
{
}

void 
CAC3Decoder::reset(acmod_t _acmod_out, const int ch_order[NCHANNELS], sample_t _level, sample_t _bias)
{
  bufptr = buf;
  bufpos = buf + 7;

  level = _level;
  bias  = _bias;
  acmod_out = _acmod_out;
  mixer.set_output_mode(acmod_out, ch_order);
  mixer.level = level;
  mixer.clev = 1.0;
  mixer.slev = 1.0;
}

// used in syncinfo, frame;
const uint8_t halfrate[12] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3};
const int rate[] =   { 32,  40,  48,  56,  64,  80,  96, 112,
                       128, 160, 192, 224, 256, 320, 384, 448,
                       512, 576, 640 };

int       
CAC3Parser::syncinfo2(uint8_t * buf)
{
  int frmsizecod;

  if ((buf[0] != 0x0b) || (buf[1] != 0x77))      // 'syncword' - syncronisation word
    return 0;
  
  if (buf[5] >= 0x60)                            // 'bsid' - bit stream identfication
    return 0;
  
  frmsizecod = buf[4] & 0x3f;
  if (frmsizecod > 0x25)
    return 0;

  switch (buf[4] & 0xc0) 
  {
    case 0:    return 4 * rate[frmsizecod >> 1];
    case 0x40: return 2 * (320 * rate[frmsizecod >> 1] / 147 + (frmsizecod & 1));
    case 0x80: return 6 * rate[frmsizecod >> 1];
    default:   return 0;
  }
}

int 
CAC3Parser::bsi2(BSI &bsi)
{

  /////////////////////////////////////////////////////////////
  // Parse bit stream information (BSI)
  /////////////////////////////////////////////////////////////
  int frmsizecod, frmsize, bsid;
  int half;

  bitstream.get(16*2); // skip syncword & crc1
  fscod      = bistram.get(2);
  frmsizecod = bitstream.get(6);
  bsid       = bitstream.get(5);
  half       = halfrate_tbl[bsid];

  bsi.bsid     = bsid;
  bsi.famesize = rate[frmsizecod >> 1];
  bsi.bitrate  = 1000 * bsi.famesize >> half;
  switch (fscod)
  {
    case 0: 
      bsi.rate = 48000 >> half;
      bsi.famesize *= 4;
      break;
    case 1:
      bsi.rate = 44100 >> half;
      bsi.famesize = (320 * bsi.frmesize / 147 + (frmsizecod & 1))
      break;
    case 2:
      bsi.rate = 32000 >> half;
      bsi.famesize *= 6;
      break;
  }

  bitstream.get(3);                    // skip bsmod
  acmod = bitstream.get (3);           // 'acmod' - audio coding mode

  if ((acmod & 1) && (acmod != 1))
    bsi.clev = clev_tbl[bitstream.get(2)]; // 'cmixlev' - center mix level

  if (acmod & 4)
    bsi.slev = slev_tbl[bitstream.get(2)]; // 'surmixlev' - surround mix level

  if ((MODE_ACMOD(bsi.acmod) == 2) && (bitstream.get (2) == 2))
    bsi.acmod |= MODE_DOLBY;

  bsi.lfeon = bitstream.get (1);           // 'lfeon' flag shows if it is LFE channel in stream
  if (lfeon)                           
    acmod |= MODE_LFE;

  dialnorm  = bitstream.get (5);       // 'dialnorm' - dialog normalization

  compre    = bitstream.get (1);       // 'compre' - compression gain word
  if (compre)
    compr   = bitstream.get (8);       // 'compr' - compression gain word

  langcode  = bitstream.get (1);       // 'langcode' - language code exists
  if (langcode)
    langcod = bitstream.get (8);       // 'langcod' - language code

  audprodie = bitstream.get (1);       // 'audprodie' - audio production information exists
  if (audprodie)
  {
    mixlevel = bitstream.get(5) + 80;  // 'mixlevel' - mixing level in SPL
    roomtyp  = bitstream.get(2);       // 'roomtyp' - room type
  }

  if (MODE_ACMOD(acmod) == MODE_DUAL)
  {
    dialnorm  = bitstream.get (5);     // 'dialnorm2' - dialog normalization

    compre    = bitstream.get (1);     // 'compr2e' - compression gain word
    if (compre)
      compr   = bitstream.get (8);     // 'compr2' - compression gain word

    langcode  = bitstream.get (1);     // 'langcod2e' - language code exists
    if (langcod)
      langcod = bitstream.get (8);     // 'langcod2' - language code

    audprodie = bitstream.get (1);     // 'audprodi2e' - audio production information exists
    if (audprodie)
    {
      mixlevel = bitstream.get(5) + 80;// 'mixlevel2' - mixing level in SPL
      roomtyp  = bitstream.get(2);     // 'roomtyp2' - room type
    }
  }

  copyrightb = bitstream.get(1);       // 'copyrightb' - copyright bit
  origbs = bitstream.get (1);          // 'origbs' - original bitstream

  if (bitstream.get (1))               // 'timecod1e' - timecode first half exists
  {
    timecode.hours = bitstream.get(5);
    timecode.mins  = bitstream.get(6);
    timecode.secs  = bitstream.get(3) << 4;
  }
  if (bitstream.get (1))               // 'timecod2e' - timecode second half exists
  {
    timecode.secs  += bitstream.get(3);
    timecode.frames = bitstream.get(5);
    timecode.fracs  = bitstream.get(6);
  }
  
  if (bitstream.get (1))               // 'addbsie' - additional bitstream information exists
  {
    int addbsil = bitstream.get (6);   // 'addbsil' - additioanl bitstream information length
    do 
    {
      bitstream.get (8);               // 'addbsi' - additional bitstream information
    } while (addbsil--);
  }

}

int       
CAC3Decoder::syncinfo (uint8_t * buf)
{
  const int rate[] = { 32,  40,  48,  56,  64,  80,  96, 112,
                       128, 160, 192, 224, 256, 320, 384, 448,
                       512, 576, 640};

  // lfe_mask[acmod] applied to 6th byte
  const uint8_t lfeon_tbl[8] = {0x10, 0x10, 0x04, 0x04, 0x04, 0x01, 0x04, 0x01};

  int frmsizecod, frmsize;
//  int bitrate;
  int half;
//  int acmod;

  // 'syncword' - syncronisation word
  if ((buf[0] != 0x0b) || (buf[1] != 0x77))
    return 0;

  // 'bsid' - bit stream identfication, high 5 bit
  // we do not understand bsid >= 12
  if (buf[5] >= 0x60)
    return 0;

  // 'bsmod' - bit stream mode
  // not used - info only field?

  // <>0 only for bsid > 8
  half = halfrate[buf[5] >> 3];

  // 'acmod' - audio coding mode
  // defines channel configuration see tbl5.7 p34
  acmod = buf[6] >> 5;
  // 'dsurmod' -  and lfeon
  if ((buf[6] & 0xf8) == 0x50)
    acmod |= MODE_DOLBY;
  if (buf[6] & lfeon_tbl[acmod])
    acmod |= MODE_LFE;
//  *flags = acmod;
//             (((buf[6] & 0xf8) == 0x50) ? acmod | MODE_DOLBY : acmod) |
//             ((buf[6] & lfeon[acmod]) ? MODE_LFE : 0);
  
  frmsizecod = buf[4] & 0x3f;
  if (frmsizecod > 0x25)
    return 0;

  frmsize = rate [frmsizecod >> 1];
  bitrate = (frmsize * 1000) >> half;

  switch (buf[4] & 0xc0) 
  {
    case 0:    sample_rate = 48000 >> half;  
      return 4 * frmsize;
    case 0x40: sample_rate = 44100 >> half;  
      return 2 * (320 * frmsize / 147 + (frmsizecod & 1));
    case 0x80: sample_rate = 32000 >> half;  
      return 6 * frmsize;
    default:
      return 0;
  }
}

int 
CAC3Decoder::frame(uint8_t * buf)
{
  const sample_t clev_tbl[4] = {LEVEL_3DB, LEVEL_45DB, LEVEL_6DB, LEVEL_45DB};
  const sample_t slev_tbl[4] = {LEVEL_3DB, LEVEL_6DB, 0, LEVEL_6DB};

  int chaninfo;

  /////////////////////////////////////////////////////////////
  // Parse bit stream
  /////////////////////////////////////////////////////////////

  state->fscod = buf[4] >> 6;
  bsid  = buf[5] >> 3;
  state->halfrate = halfrate[bsid];
  acmod = buf[6] >> 5;

  // reset bitsteram pointer
  bitstream.set_ptr (buf + 6);

  // skip acmod already have
  bitstream.get (3);

  // if 2/0 mode it is 'dsurmod' flag that shows that stream
  // is Dolby surround coded. It is not used by decoder
  // but may be used by other programs
  if ((MODE_ACMOD(acmod) == 2) && (bitstream.get (2) == 2))
    acmod |= MODE_DOLBY;

  // if 3/x mode it is 'cmixlev' - center mix level value  
  // which choose center downmix level 'clev' in dB
  if ((acmod & 1) && (acmod != 1))
    clev = clev_tbl[bitstream.get (2)];

  // if surround modes it is 'surmixlev' - surround mix level value
  // which choose surround downmix level 'slev' in dB
  if (acmod & 4)
    slev = slev_tbl[bitstream.get (2)];

  // 'lfeon' flag shows if it is LFE channel in stream
  lfeon = bitstream.get (1);
  if (lfeon)
    acmod |= MODE_LFE;

  /////////////////////////////////////////////////////////////
  // Skip everything we do not need
  /////////////////////////////////////////////////////////////

  chaninfo = !acmod;
  do 
  {
    bitstream.get (5);     /* dialnorm */
    if (bitstream.get (1)) /* compre */
      bitstream.get (8);   /* compr */
    if (bitstream.get (1)) /* langcode */
      bitstream.get (8);   /* langcod */
    if (bitstream.get (1)) /* audprodie */
      bitstream.get (7);   /* mixlevel + roomtyp */
  } 
  while (chaninfo--);

  bitstream.get (2);       /* copyrightb + origbs */

  if (bitstream.get (1))   /* timecod1e */
    bitstream.get (14);    /* timecod1 */
  if (bitstream.get (1))   /* timecod2e */
    bitstream.get (14);    /* timecod2 */

  if (bitstream.get (1))   /* addbsie */
  {
    int addbsil;
    addbsil = bitstream.get (6);
    do {
      bitstream.get (8);   /* addbsi */
    } while (addbsil--);
  }

  /////////////////////////////////////////////////////////////
  // Init variables to for first block decoding
  /////////////////////////////////////////////////////////////

//  // the 2* compensates for differences in imdct 
//  state->dynrng = 2 * level;
//  state->dynrnge = 1;
//  state->dynrngcall = 0;
  state->cplba.deltbae = DELTA_BIT_NONE;
  state->ba[0].deltbae = 
    state->ba[1].deltbae = state->ba[2].deltbae =
    state->ba[3].deltbae = state->ba[4].deltbae = DELTA_BIT_NONE;

  return 0;
}

int 
CAC3Decoder::parse_block()
{
  // nfchans[acmod] - number of fbw (full-bandwidth) channels
  int   nfchans;
//  const uint8_t nfchans_tbl[] = {2, 1, 2, 3, 3, 4, 4, 5, 1, 1, 2};

  // block switch and dither flags for each fbw channel
  uint8_t blksw[5], dithflag[5];

  const int rematrix_band[4] = {25, 37, 61, 253};
  int i, chaninfo;
  uint8_t cplexpstr, chexpstr[5], lfeexpstr, do_bit_alloc, done_cpl;
  quantizer_t quantizer;



  /////////////////////////////////////////////////////////////
  // Parse bitstream
  /////////////////////////////////////////////////////////////

  // get number of fbw channels
  nfchans = MODE_NFCHANS(acmod);

  // get block switch flags
  for (i = 0; i < nfchans; i++)
    blksw[i] = bitstream.get (1);

  // get dither flags
  for (i = 0; i < nfchans; i++)
    dithflag[i] = bitstream.get (1);

  // dynamic range control values
  chaninfo = !acmod; // if double-mono mode do following twice
  do 
  { 
    // 'dynrnge' - dynamic range gain exists
    if (dynrnge = (bitstream.get (1) == 1)) 
    {  
      // 'dynrng' - dynamic range gain word
      int dynrng = bitstream.get_2(8);
      dynrng_gain = ((((dynrng & 0x1f) | 0x20) << 13) * scale_factor[3 - (dynrng >> 5)]);
    }
  } while (chaninfo--);

  // coupling strategy information
  // 'cplstre' - coupling strategy exists
  if (bitstream.get (1)) 
  {
    state->chincpl = 0;

    // 'cplinu' - coupling in use
    if (bitstream.get (1)) 
    {
      const uint8_t bndtab[16] = {31, 35, 37, 39, 41, 42, 43, 44,
                                  45, 45, 46, 46, 47, 47, 48, 48};
      int cplbegf;
      int cplendf;
      int ncplsubnd;

      // 'chincpl' - channel in coupling flag for each channel
      // couple them in one word
      // maybe some optimizations here?
	    for (i = 0; i < nfchans; i++)
        state->chincpl |= bitstream.get (1) << i;

      // check for allowed modes for coupling
      // it is prohibited in mono and dual-mono modes
      switch (MODE_ACMOD(acmod))
      {
        case 0: 
        case 1:
          return 1;

        case 2:
          // 'phsflginu' - phase flag in use, only for 2/0 mode
          // mean that phase flags are with coupling coord info
          state->phsflginu = bitstream.get (1);
	    }

      // 'cplbegf' & 'cplendf' coupling begin/end frequency code
      // interpreterd as sub-band numbers
      cplbegf = bitstream.get (4);
      cplendf = bitstream.get (4);

      // check if the end band is before begin
      if (cplendf + 3 - cplbegf < 0)
        return 1;

      // calc 'ncplbnd' - number of sub-bands in coupling
      state->ncplbnd = ncplsubnd = cplendf + 3 - cplbegf;

      state->cplstrtbnd = bndtab[cplbegf];
      state->cplstrtmant = cplbegf * 12 + 37;
      state->cplendmant = cplendf * 12 + 73;

      state->cplbndstrc = 0;

      // for each coupling sub-band get 'cplbndstrc' - coupling
      // band structure. bands are combined into wider bands
      // 0 means new wide sub-band and 1 means that this sub-band
      // is combined with previous sub-band into a wider band
      for (i = 0; i < ncplsubnd - 1; i++)
        if (bitstream.get (1)) 
        {
          state->cplbndstrc |= 1 << i;
          state->ncplbnd--;
        }
    }
  } // coupling information

  // get coupling coordinates for each channel.
  // only if 'cplinu', but in this case it is must
  // be 'chincpl' - channels in coupling? (nothing about it in standart)
  if (state->chincpl) 
  {
    int j;
    int cplcoe = 0;

    // get coupling coordinates for each channel
    for (i = 0; i < nfchans; i++)
      // if this channel is in coupling
      if ((state->chincpl) >> i & 1)
        // if 'cplcoe' - coupling coorinates exists
        if (bitstream.get (1)) 
        {
          int mstrcplco, cplcoexp, cplcomant;
          cplcoe = 1;
          // 'mstrcplco' - master coupling coordinate
          mstrcplco = 3 * bitstream.get (2);

          // for each coupling band
          for (j = 0; j < state->ncplbnd; j++) 
          {
            // 'cplcoexp' - coupling coordinate exponent
            cplcoexp = bitstream.get (4);
            // 'cplcomant' - coupling coordinate mantissa
            cplcomant = bitstream.get (4);

            if (cplcoexp == 15)
              cplcomant <<= 14;
            else
              cplcomant = (cplcomant | 0x10) << 13;

            state->cplco[i][j] =
              cplcomant * scale_factor[cplcoexp + mstrcplco];
          }
        }
    // if 2/x mode & coupling & phase flag in use then
    // read read phase flags for each band
    if ((MODE_ACMOD(acmod) == MODE_STEREO) && state->phsflginu && cplcoe)
      for (j = 0; j < state->ncplbnd; j++)
        // 'phsflg' - phase flag
        // whether decoder should phase invert right channel mantissas
        if (bitstream.get (1))
          state->cplco[1][j] = -state->cplco[1][j];
  }

  // rematrixing strategy information for each band
  // rematrixing strategy applies only at 2/0 mode
  // 'rematstr' - rematrixing strategy
  // if new rematrixing information present in this block
  if ((MODE_ACMOD(acmod) == MODE_STEREO) && (bitstream.get (1))) 
  {
    int end;

    state->rematflg = 0;

    // use 'chincpl' instead 'cplinu' again
    end = (state->chincpl) ? state->cplstrtmant : 253;	
    i = 0;
    do
      // 'rematflg' - rematrix flag - if rematrixing is applied
      state->rematflg |= bitstream.get (1) << i;
    while (rematrix_band[i++] < end);
  }

  cplexpstr = EXP_REUSE;
  lfeexpstr = EXP_REUSE;

  // 'cplexpstr' - coupling exponent strategy
  // only if 'cplinu' - coupling in use
  // use 'chincpl' instead of 'cplinu' again
  if (state->chincpl) 
    cplexpstr = bitstream.get (2);

  // 'chexpstr' - channel exponent strategy for each fbw channel
  for (i = 0; i < nfchans; i++)
    chexpstr[i] = bitstream.get (2);

  // 'lfeexpstr' - LFE exponent strategy
  // only if 'lfeon'
  if (lfeon) 
    lfeexpstr = bitstream.get (1);

  // load or reuse channel bandwidth for each fbw channel
  for (i = 0; i < nfchans; i++)
    if (chexpstr[i] != EXP_REUSE) 
    // if not to reuse old value
    {

      if ((state->chincpl >> i) & 1)
        // if coupled channel then use ...
        state->endmant[i] = state->cplstrtmant;
      else 
      {
        // 'chbwcod' = channel bandwidth code
        // defines upper band edge of not coupled fbw channel
        // must be <= 60
        int chbwcod;
        chbwcod = bitstream.get (6);
        if (chbwcod > 60)
          return 1;
        state->endmant[i] = chbwcod * 3 + 73;
      }
    }

  do_bit_alloc = 0;

  // parse exponents for coupling channel
  // if not reuse old values
  if (cplexpstr != EXP_REUSE) 
  {
    int cplabsexp, ncplgrps;

    do_bit_alloc = 64;

    // 'ncplgrps' - number of coupling groups
    ncplgrps = ((state->cplendmant - state->cplstrtmant) /
                (3 << (cplexpstr - 1)));

    // 'cplabsexp' - coupling absolute exponents
    // referense exponent when decoding the differential 
    // exponents for the coupling channel
    cplabsexp = bitstream.get (4) << 1;

    // parse differrential exponents 'cplexps' for coupling channel
    if (parse_exponents (cplexpstr, ncplgrps, cplabsexp,
        state->cpl_expbap.exp + state->cplstrtmant))
      return 1;
  }

  // parse exponents for each channel
  // if not reuse old values
  for (i = 0; i < nfchans; i++)
    if (chexpstr[i] != EXP_REUSE) 
    {
      int grp_size, nchgrps;

      do_bit_alloc |= 1 << i;
      grp_size = 3 << (chexpstr[i] - 1);
      nchgrps = (state->endmant[i] + grp_size - 4) / grp_size;

      // 'exps[0]' - absolute exponent
      state->fbw_expbap[i].exp[0] = bitstream.get (4);
      // parse differrential exponents 'exps'
      if (parse_exponents (chexpstr[i], nchgrps, state->fbw_expbap[i].exp[0],
          state->fbw_expbap[i].exp + 1))
        return 1;

      // 'gainrng'  - channel gain range code
      bitstream.get (2);
    }

  // parse LFE exponents
  // if not reuse old values
  if (lfeexpstr != EXP_REUSE) 
  {
    do_bit_alloc |= 32;
    // 'lfeexps[0]' - absolute exponent
    state->lfe_expbap.exp[0] = bitstream.get (4);
    // prase 2 diffenrential exponents
    if (parse_exponents (lfeexpstr, 2, state->lfe_expbap.exp[0], 
        state->lfe_expbap.exp + 1))
      return 1;
  }

  // 'baie' - bit allication information exists
  if (bitstream.get (1)) 
  {
    do_bit_alloc = -1;
    // bit allocation information
    // 'sdcycod' - slow decay code
    // 'fdcycod' - fast decay code
    // 'sgaincod' - slow gain code
    // 'dbpbcod' - Db per bit code
    // 'floorcod' - masking floor code
    state->bai = bitstream.get (11);
  }

  // 'snroffste' - SNR offset exists
  if (bitstream.get (1)) 
  {
    do_bit_alloc = -1;
    // 'csnroffst' - coarse SNR offset
    state->csnroffst = bitstream.get (6);

    // use 'chincpl' instead of 'cplinu' again
    if (state->chincpl)
      // 'cplfsnroffst' - coupling fine SNR offset
      // 'cplfgaincod' - coupling fast gain code
      state->cplba.bai = bitstream.get (7);

    for (i = 0; i < nfchans; i++)
      // 'fsnroffst' - channel fine SNR offset
      // 'fgaincod' - channel fast gain code
      state->ba[i].bai = bitstream.get (7);

    if (lfeon)
      // 'lfesnroffst' - LFE channel SNR offset
      // 'lfegaincod' - LFE channel gain code
      state->lfeba.bai = bitstream.get (7);
  }

  // use 'chincpl' instead of 'cplinu' again
  // 'cplleake' - coupling leak initialization exists
  if ((state->chincpl) && (bitstream.get (1))) 
  {
    do_bit_alloc |= 64;
    // 'cplfleak' - coupling fast leak initialization
    state->cplfleak = 9 - bitstream.get (3);
    // 'cplsleak' - coupling slow leak initialization
    state->cplsleak = 9 - bitstream.get (3);
  }

  // 'deltbaie' - delta bit allication information exists
  if (bitstream.get (1))
  {
    do_bit_alloc = -1;

    // use 'chincpl' instead of 'cplinu' again
    if (state->chincpl)
      // 'cpldeltbae' - couling delta bit allocation information exists
      // determines delta bit allocation strategy
      state->cplba.deltbae = bitstream.get (2);

    for (i = 0; i < nfchans; i++)
      // 'deltbae' - delta bit allocation exists
      // determines delta bit allocation strategy
      state->ba[i].deltbae = bitstream.get (2);

    // use 'chincpl' instead of 'cplinu' again
    // parse delta bit information for coupling channel if needed
    if (state->chincpl && (state->cplba.deltbae == DELTA_BIT_NEW) &&
        parse_deltba (state->cplba.deltba))
      return 1;

    for (i = 0; i < nfchans; i++)
      // parse delta bit information for fbw channels
      if ((state->ba[i].deltbae == DELTA_BIT_NEW) &&
           parse_deltba (state->ba[i].deltba))
        return 1;
  }

  // perform bit allocation
  if (do_bit_alloc) 
  {
    if (zero_snr_offsets (nfchans, state)) 
    {
      memset (state->cpl_expbap.bap, 0, sizeof (state->cpl_expbap.bap));

      for (i = 0; i < nfchans; i++)
        memset (state->fbw_expbap[i].bap, 0, sizeof (state->fbw_expbap[i].bap));
        memset (state->lfe_expbap.bap, 0, sizeof (state->lfe_expbap.bap));
    } 
    else 
    {
      // use 'chincpl' instead of 'cplinu' again
      if (state->chincpl && (do_bit_alloc & 64))
        a52_bit_allocate (state, &state->cplba, state->cplstrtbnd,
                          state->cplstrtmant, state->cplendmant,
                          state->cplfleak << 8, state->cplsleak << 8,
                          &state->cpl_expbap);
      for (i = 0; i < nfchans; i++)
        if (do_bit_alloc & (1 << i))
          a52_bit_allocate (state, state->ba + i, 0, 0, state->endmant[i], 0, 0,
                            state->fbw_expbap +i);
      if (lfeon && (do_bit_alloc & 32)) 
      {
        state->lfeba.deltbae = DELTA_BIT_NONE;
        a52_bit_allocate (state, &state->lfeba, 0, 0, 7, 0, 0, &state->lfe_expbap);
      }
    }
  }

  // 'skiple' - skip length exists
  if (bitstream.get (1)) 
  {
    // 'skipl' - slkip length
    i = bitstream.get (9);
    // skip
    while (i--)
      bitstream.get (8);
  }


///////////////////////////////////////////////////////
//        HERE WE GO
///////////////////////////////////////////////////////


  quantizer.q1_ptr = quantizer.q2_ptr = quantizer.q4_ptr = -1;
  done_cpl = 0;

  for (i = 0; i < nfchans; i++)
  {
    int j;

    coeff_get (samples[i], state->fbw_expbap +i, &quantizer,
               dithflag[i], state->endmant[i]);

    if ((state->chincpl >> i) & 1) 
    {
      if (!done_cpl) 
      {
        done_cpl = 1;

        double coeff[5] = {0, 0, 0, 0, 0};

        coeff_get_coupling (nfchans, &quantizer, dithflag);
      }
      j = state->cplendmant;
    } 
    else
      j = state->endmant[i];

    do
      samples[i][j] = 0;
    while (++j < 256);
  }

  if (MODE_ACMOD(acmod) == MODE_STEREO) 
  {
    int j, end, band, rematflg;

    end = ((state->endmant[0] < state->endmant[1]) ?
            state->endmant[0] : state->endmant[1]);

    i = 0;
    j = 13;
    rematflg = state->rematflg;
    do 
    {
      if (! (rematflg & 1)) 
      {
        rematflg >>= 1;
        j = rematrix_band[i++];
        continue;
      }
      rematflg >>= 1;
      band = rematrix_band[i++];

      if (band > end)
        band = end;

      do 
      {
        sample_t tmp0, tmp1;
        tmp0 = samples[CH_L][j];
        tmp1 = samples[CH_R][j];
        samples[CH_L][j] = tmp0 + tmp1;
        samples[CH_R][j] = tmp0 - tmp1;
      } while (++j < band);

    } while (j < end);
  }

  if (lfeon) 
  {
    coeff_get (samples[CH_LFE], &state->lfe_expbap, &quantizer, 0, 7);
    for (i = 7; i < 256; i++)
      samples[CH_LFE][i] = 0;
    imdct.imdct_512 (samples[CH_LFE], samples1[CH_LFE], bias);
  }


  for (i = 0; i < nfchans; i++) 
  {
    if (blksw[i])
      imdct.imdct_256 (samples[i], samples1[i], bias);
    else 
     imdct.imdct_512 (samples[i], samples1[i], bias);
  }

  return 0;
}







void
CAC3Decoder::decode(uint8_t *buffer, uint32_t length)
{
  long len;

  // while we have something in buffer we will try to decode it
  while (1) 
  {
    // copy requested data to buf from buffer if we have requested amount
    // or copy rest of buffer to remember it for future calls so we have processed all given data

    len = length;

    if (!len) break;

    if (len > bufpos - bufptr)
      len = bufpos - bufptr;

    memcpy (bufptr, buffer, len);
    bufptr += len;
    buffer += len;
    length -= len;

    // if we have all requested data
    if (bufptr == bufpos) 
    {
      // find sync, decode frame header and load frame data
      if (bufpos == buf + 7) 
      {
        int frame_size = syncinfo (buf);

        // skip if it is not a sync and cycle
        if (!frame_size) 
        {
          for (bufptr = buf; bufptr < buf + 6; bufptr++)
             bufptr[0] = bufptr[1];
          continue;
        }

        // or request more data
        bufpos = buf + frame_size;
      } 
      // decode frame data
      else 
      {
        sample_t _level = level;
        if (!frame (buf))
        {
          mixer.set_input_mode(acmod);

          for (int i = 0; i < 6; i++) 
          {
            if (parse_block()) break;

            if (dynrng && dynrnge)
            {
              mixer.dynrng = dynrng_gain;
              mixer.calc_matrix();
            }
            if (!dynrng)
            {
              mixer.dynrng = 1.0;
              mixer.calc_matrix();
            }

            mixer.mix(samples);

            block(samples);
          }
        }        
        bufptr = buf;
        bufpos = buf + 7;
      }
    }
  }
}
