#include <windows.h>
#include <winreg.h>
#include <stdio.h>
#include "ac3decoder.h"
#include "registry.h"


///////////////////////////////////////////////////////////////////////////////
// Decoder Control
///////////////////////////////////////////////////////////////////////////////

DecoderControl::DecoderControl()
{
  thread = 0;
  reset_time();
}

DecoderControl::~DecoderControl()
{}

void 
DecoderControl::reset()
{
  thread = 0;
  reset_time();
  Decoder::reset();
}


STDMETHODIMP DecoderControl::load_params(const char *_preset)
{
  acmod_t acmod;
  char matrix[256];
  bool dynrng;
  bool slev_lock, clev_lock, lfelev_lock;
  bool auto_matrix;
  bool normalize_matrix;
  bool normalize;
  bool expand_stereo;
  bool voice_control;
  sample_t dynrng_power;
  sample_t master, slev, clev, lfelev;

  // open registry

  char preset[256];
  if (_preset)
    sprintf(preset, "Software\\AC3Filter\\preset\\%s", _preset);
  else
    sprintf(preset, "Software\\AC3Filter\\preset\\_default");

  RegistryKey reg(preset);

  // if registry key does not exists then load defaults
  acmod             = reg.get_int32("acmod", MODE_ERROR);
  dynrng            = reg.get_bool("dynrng", true);
  auto_matrix       = reg.get_bool("auto_matrix", true);
  normalize_matrix  = reg.get_bool("normalize_matrix", false);
  normalize         = reg.get_bool("normalize", true);
  expand_stereo     = reg.get_bool("expand_stereo", true);
  voice_control     = reg.get_bool("voice_control", true);
  clev_lock         = reg.get_bool("clev_lock", true);
  slev_lock         = reg.get_bool("slev_lock", true);
  lfelev_lock       = reg.get_bool("lfelev_lock", true);
  clev              = reg.get_double("clev", 1.0);
  slev              = reg.get_double("slev", 1.0);
  lfelev            = reg.get_double("lfelev", 1.0);
  master            = reg.get_double("master", 1.0);
  dynrng_power      = reg.get_double("dynrng_power", 1.0);

  if (!IS_MODE_ERROR(acmod))
    set_speakers(acmod);

  if (reg.get_text("matrix", matrix, 256))
    load_matrix(matrix);

  set_dynrng(dynrng, dynrng_power);
  set_bsi_locks(slev_lock, clev_lock, lfelev_lock);
  set_gain(master);
  set_gains(slev, clev, lfelev);
  set_normalize(normalize);
  set_normalize_matrix(normalize_matrix);
  set_expand_stereo(expand_stereo);
  set_voice_control(voice_control);
  set_auto_matrix(auto_matrix);

  return S_OK;
}

STDMETHODIMP DecoderControl::save_params(const char *_preset)
{
  acmod_t acmod;
  bool dynrng;
  bool slev_lock, clev_lock, lfelev_lock;
  bool auto_matrix;
  bool normalize_matrix;
  bool normalize;
  bool expand_stereo, voice_control;
  sample_t dynrng_power, dynrng_level;
  sample_t master, gain, slev, clev, lfelev;


  // open registry

  char preset[256];
  if (_preset)
    sprintf(preset, "Software\\AC3Filter\\preset\\%s", _preset);
  else
    sprintf(preset, "Software\\AC3Filter\\preset\\_default");

  RegistryKey reg(preset);
  if (!reg.is_ok())
  {
    // try to create key for new preset
    reg.create_key("Software", "AC3Filter");
    reg.create_key("Software\\AC3Filter", "preset");
    if (_preset)
      reg.create_key("Software\\AC3Filter\\preset", _preset);
    else
      reg.create_key("Software\\AC3Filter\\preset", "_default");
  }

  if (!reg.is_ok())
    return S_FALSE;

  // load data to save

  get_speakers(&acmod);
  get_dynrng(&dynrng, &dynrng_level, &dynrng_power);
  get_bsi_locks(&slev_lock, &clev_lock, &lfelev_lock);
  get_gain(&master, &gain);
  get_gains(&slev, &clev, &lfelev);
  get_normalize(&normalize);
  get_normalize_matrix(&normalize_matrix);
  get_auto_matrix(&auto_matrix);
  get_expand_stereo(&expand_stereo);
  get_voice_control(&voice_control);

  // save data

  reg.set_int32 ("acmod"            ,acmod           );
  reg.set_bool  ("dynrng"           ,dynrng          );
  reg.set_bool  ("normalize"        ,normalize       );
  reg.set_bool  ("normalize_matrix" ,normalize_matrix);
  reg.set_bool  ("auto_matrix"      ,auto_matrix     );
  reg.set_bool  ("expand_stereo"    ,expand_stereo   );
  reg.set_bool  ("voice_control"    ,voice_control   );
  reg.set_bool  ("clev_lock"        ,clev_lock       );
  reg.set_bool  ("slev_lock"        ,slev_lock       );
  reg.set_bool  ("lfelev_lock"      ,lfelev_lock     );
  reg.set_double("clev"             ,clev            );
  reg.set_double("slev"             ,slev            );
  reg.set_double("lfelev"           ,lfelev          );
  reg.set_double("master"           ,master          );
  reg.set_double("dynrng_power"     ,dynrng_power    );

  return S_OK;
}

STDMETHODIMP DecoderControl::load_matrix(const char *_preset)
{
  mixer_matrix_t matrix;

  // open registry

  char preset[256];
  if (_preset)
    sprintf(preset, "Software\\AC3Filter\\matrix\\%s", _preset);
  else
    sprintf(preset, "Software\\AC3Filter\\matrix\\_default");

  RegistryKey reg(preset);

  matrix[0][0] = reg.get_double("L_L",   1.0);
  matrix[0][1] = reg.get_double("L_C",   0.0);
  matrix[0][2] = reg.get_double("L_R",   0.0);
  matrix[0][3] = reg.get_double("L_SL",  0.0);
  matrix[0][4] = reg.get_double("L_SR",  0.0);
  matrix[0][5] = reg.get_double("L_LFE", 0.0);

  matrix[1][0] = reg.get_double("C_L",   0.0);
  matrix[1][1] = reg.get_double("C_C",   1.0);
  matrix[1][2] = reg.get_double("C_R",   0.0);
  matrix[1][3] = reg.get_double("C_SL",  0.0);
  matrix[1][4] = reg.get_double("C_SR",  0.0);
  matrix[1][5] = reg.get_double("C_LFE", 0.0);

  matrix[2][0] = reg.get_double("R_L",   0.0);
  matrix[2][1] = reg.get_double("R_C",   0.0);
  matrix[2][2] = reg.get_double("R_R",   1.0);
  matrix[2][3] = reg.get_double("R_SL",  0.0);
  matrix[2][4] = reg.get_double("R_SR",  0.0);
  matrix[2][5] = reg.get_double("R_LFE", 0.0);

  matrix[3][0] = reg.get_double("SL_L",   0.0);
  matrix[3][1] = reg.get_double("SL_C",   0.0);
  matrix[3][2] = reg.get_double("SL_R",   0.0);
  matrix[3][3] = reg.get_double("SL_SL",  1.0);
  matrix[3][4] = reg.get_double("SL_SR",  0.0);
  matrix[3][5] = reg.get_double("SL_LFE", 0.0);

  matrix[4][0] = reg.get_double("SR_L",   0.0);
  matrix[4][1] = reg.get_double("SR_C",   0.0);
  matrix[4][2] = reg.get_double("SR_R",   0.0);
  matrix[4][3] = reg.get_double("SR_SL",  0.0);
  matrix[4][4] = reg.get_double("SR_SR",  1.0);
  matrix[4][5] = reg.get_double("SR_LFE", 0.0);

  matrix[5][0] = reg.get_double("LFE_L",   0.0);
  matrix[5][1] = reg.get_double("LFE_C",   0.0);
  matrix[5][2] = reg.get_double("LFE_R",   0.0);
  matrix[5][3] = reg.get_double("LFE_SL",  0.0);
  matrix[5][4] = reg.get_double("LFE_SR",  0.0);
  matrix[5][5] = reg.get_double("LFE_LFE", 1.0);

  set_matrix(&matrix);

  return S_OK;
}

STDMETHODIMP DecoderControl::save_matrix(const char *_preset)
{
  mixer_matrix_t matrix;
  get_matrix(&matrix);

  // open registry

  char preset[256];
  if (_preset)
    sprintf(preset, "Software\\AC3Filter\\matrix\\%s", _preset);
  else
    sprintf(preset, "Software\\AC3Filter\\matrix\\_default");

  RegistryKey reg(preset);
  if (!reg.is_ok())
  {
    // try to create key for new preset
    reg.create_key("Software", "AC3Filter");
    reg.create_key("Software\\AC3Filter", "matrix");
    if (_preset)
      reg.create_key("Software\\AC3Filter\\matrix", _preset);
    else
      reg.create_key("Software\\AC3Filter\\matrix", "_default");
  }

  if (!reg.is_ok())
    return S_FALSE;



  reg.set_double("L_L",    matrix[0][0]);
  reg.set_double("L_C",    matrix[0][1]);
  reg.set_double("L_R",    matrix[0][2]);
  reg.set_double("L_SL",   matrix[0][3]);
  reg.set_double("L_SR",   matrix[0][4]);
  reg.set_double("L_LFE",  matrix[0][5]);
                                       
  reg.set_double("C_L",    matrix[1][0]);
  reg.set_double("C_C",    matrix[1][1]);
  reg.set_double("C_R",    matrix[1][2]);
  reg.set_double("C_SL",   matrix[1][3]);
  reg.set_double("C_SR",   matrix[1][4]);
  reg.set_double("C_LFE",  matrix[1][5]);
                                       
  reg.set_double("R_L",    matrix[2][0]);
  reg.set_double("R_C",    matrix[2][1]);
  reg.set_double("R_R",    matrix[2][2]);
  reg.set_double("R_SL",   matrix[2][3]);
  reg.set_double("R_SR",   matrix[2][4]);
  reg.set_double("R_LFE",  matrix[2][5]);
      
  reg.set_double("SL_L",   matrix[3][0]);
  reg.set_double("SL_C",   matrix[3][1]);
  reg.set_double("SL_R",   matrix[3][2]);
  reg.set_double("SL_SL",  matrix[3][3]);
  reg.set_double("SL_SR",  matrix[3][4]);
  reg.set_double("SL_LFE", matrix[3][5]);
                                       
  reg.set_double("SR_L",   matrix[4][0]);
  reg.set_double("SR_C",   matrix[4][1]);
  reg.set_double("SR_R",   matrix[4][2]);
  reg.set_double("SR_SL",  matrix[4][3]);
  reg.set_double("SR_SR",  matrix[4][4]);
  reg.set_double("SR_LFE", matrix[4][5]);

  reg.set_double("LFE_L",  matrix[5][0]);
  reg.set_double("LFE_C",  matrix[5][1]);
  reg.set_double("LFE_R",  matrix[5][2]);
  reg.set_double("LFE_SL", matrix[5][3]);
  reg.set_double("LFE_SR", matrix[5][4]);
  reg.set_double("LFE_LFE",matrix[5][5]);

  return S_OK;
}

STDMETHODIMP DecoderControl::load_equalizer(const char *_preset)
{
  return S_OK;
}

STDMETHODIMP DecoderControl::save_equalizer(const char *_preset)
{
  return S_OK;
}



void 
DecoderControl::reset_time()
{
  __int64 creation_time;
  __int64 exit_time;
  __int64 kernel_time;
  __int64 user_time;

  SYSTEMTIME systime;

  GetSystemTime(&systime);
  SystemTimeToFileTime(&systime, (FILETIME*)&system_time_begin);

  if (thread)
  {
    if (GetThreadTimes(thread, 
           (FILETIME*)&creation_time, 
           (FILETIME*)&exit_time, 
           (FILETIME*)&kernel_time, 
           (FILETIME*)&user_time))
    {
      thread_time = 0;
      thread_time_begin = kernel_time + user_time;
    }
    else
    {
      thread_time = 0;
      thread_time_begin = 0;
    }
  }
}

void 
DecoderControl::decode(uint8_t *buffer, uint32_t length)
{
  DuplicateHandle(GetCurrentProcess(), GetCurrentThread(), GetCurrentProcess(), &thread, 0, true, DUPLICATE_SAME_ACCESS);
  __int64 creation_time;
  __int64 exit_time;
  __int64 kernel_time;
  __int64 user_time;

  if (GetThreadTimes(thread, 
         (FILETIME*)&creation_time, 
         (FILETIME*)&exit_time, 
         (FILETIME*)&kernel_time, 
         (FILETIME*)&user_time))
    thread_time_begin = kernel_time + user_time;

  Decoder::decode(buffer, length);

  if (GetThreadTimes(thread, 
         (FILETIME*)&creation_time, 
         (FILETIME*)&exit_time, 
         (FILETIME*)&kernel_time, 
         (FILETIME*)&user_time))
    thread_time += kernel_time + user_time - thread_time_begin;

  CloseHandle(thread);
  thread = 0;
  thread_time_begin = 0;
}

STDMETHODIMP
DecoderControl::get_cpu_load(double *cpu_load)
{
  __int64 creation_time;
  __int64 exit_time;
  __int64 kernel_time;
  __int64 user_time;
  __int64 system_time_end;
  SYSTEMTIME systime;

  GetSystemTime(&systime);
  SystemTimeToFileTime(&systime, (FILETIME *)&system_time_end);


  if (thread)
  {
    if (GetThreadTimes(thread, 
           (FILETIME*)&creation_time, 
           (FILETIME*)&exit_time, 
           (FILETIME*)&kernel_time, 
           (FILETIME*)&user_time))
    {
      thread_time += kernel_time + user_time - thread_time_begin;
      thread_time_begin = kernel_time + user_time;
    }
    else
    {
      thread_time = 0;
      thread_time_begin = 0;
    }
  }

  if (system_time_begin - system_time_end)
    *cpu_load = double(thread_time) / double(system_time_end - system_time_begin);
  else
    *cpu_load = 0;

  thread_time = 0;
  system_time_begin = system_time_end;
  return S_OK;
}



STDMETHODIMP DecoderControl::get_bsi(BSI *_bsi)
{
  memcpy(_bsi, &bsi, sizeof(bsi));
  return S_OK;
}

STDMETHODIMP DecoderControl::get_speakers(acmod_t *_speakers)
{
  *_speakers = get_acmod();
  return S_OK;
}


STDMETHODIMP DecoderControl::get_stat(int *_frames, int *_errors)                                                        
{
  *_frames = frames;
  *_errors = errors;
  return S_OK;
}


// gain control
STDMETHODIMP DecoderControl::get_levels(sample_t *_source, sample_t *_max_source, sample_t *_speaker, sample_t *_max_speaker)
{
  *_max_source  = 1.0;
  *_max_speaker = mixer.level;
  memcpy(_source,  &mixer.in_levels, sizeof(mixer.in_levels));
  memcpy(_speaker, &mixer.out_levels, sizeof(mixer.out_levels));
  memset(&mixer.in_levels, 0, sizeof(mixer.in_levels));
  memset(&mixer.out_levels, 0, sizeof(mixer.out_levels));
  return S_OK;
}

STDMETHODIMP DecoderControl::get_gain(sample_t *_master, sample_t *_gain)
{
  *_master = mixer.master;
  *_gain   = mixer.gain;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_gain(sample_t _master)
{
  mixer.master = _master;
  mixer.gain = _master;
  return S_OK;
}

STDMETHODIMP DecoderControl::get_dynrng(bool *_dynrng, sample_t *_dynrng_level, sample_t *_dynrng_power)
{
  *_dynrng = dynrng;
  *_dynrng_level = mixer.dynrng;
  *_dynrng_power = dynrng_power;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_dynrng(bool _dynrng, sample_t _dynrng_power)
{
  dynrng = _dynrng;
  dynrng_power = _dynrng_power;
  return S_OK;
}


// mixer
STDMETHODIMP DecoderControl::get_matrix(mixer_matrix_t *_matrix)                                    
{
  memcpy(_matrix, mixer.matrix, sizeof(mixer_matrix_t));
  return S_OK;
}

STDMETHODIMP DecoderControl::set_matrix(mixer_matrix_t *_matrix)                                                           
{
  memcpy(mixer.matrix, _matrix, sizeof(mixer_matrix_t));
  return S_OK;
}

STDMETHODIMP DecoderControl::get_gains(sample_t *_slev, sample_t *_clev, sample_t *_lfelev)                                 
{
  *_slev   = mixer.slev;
  *_clev   = mixer.clev;
  *_lfelev = mixer.lfelev;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_gains(sample_t _slev, sample_t _clev, sample_t _lfelev)                                 
{
  mixer.slev   = _slev;
  mixer.clev   = _clev;
  mixer.lfelev = _lfelev;
  return S_OK;
}

STDMETHODIMP DecoderControl::get_bsi_locks(bool *_slev_lock, bool *_clev_lock, bool *_lfe_lock)                                 
{
  *_slev_lock = slev_lock;
  *_clev_lock = clev_lock;
  *_lfe_lock  = lfelev_lock;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_bsi_locks(bool _slev_lock, bool _clev_lock, bool _lfe_lock)                                 
{
  slev_lock   = _slev_lock;
  clev_lock   = _clev_lock;
  lfelev_lock = _lfe_lock;
  return S_OK;
}

STDMETHODIMP DecoderControl::get_auto_matrix(bool *_auto_matrix)                                                              
{
  *_auto_matrix = auto_matrix;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_auto_matrix(bool _auto_matrix)                                                              
{
  auto_matrix = _auto_matrix;
  return S_OK;
}

STDMETHODIMP DecoderControl::get_normalize_matrix(bool *_normalize_matrix)                                                                  
{
  *_normalize_matrix = normalize_matrix;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_normalize_matrix(bool _normalize_matrix)                                                                  
{
  normalize_matrix = _normalize_matrix;
  return S_OK;
}

STDMETHODIMP DecoderControl::get_normalize(bool *_normalize)                                                                  
{
  *_normalize = mixer.normalize;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_normalize(bool _normalize)                                                                  
{
  mixer.normalize = _normalize;
  return S_OK;
}

STDMETHODIMP DecoderControl::get_expand_stereo(bool *_expand_stereo)
{
  *_expand_stereo = mixer.expand_stereo;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_expand_stereo(bool  _expand_stereo)
{
  mixer.expand_stereo = _expand_stereo;
  return S_OK;
}

STDMETHODIMP DecoderControl::get_voice_control(bool *_voice_control)
{
  *_voice_control = mixer.voice_control;
  return S_OK;
}

STDMETHODIMP DecoderControl::set_voice_control(bool  _voice_control)
{
  mixer.voice_control = _voice_control;
  return S_OK;
}

                                           
// equalizer
STDMETHODIMP DecoderControl::get_equalize(sample_buffer_t *func)                                                                    
{  
  return S_OK;
}

STDMETHODIMP DecoderControl::set_equalize(sample_buffer_t *func)                                                                    
{
  return S_OK;
}

STDMETHODIMP DecoderControl::get_spectrum(int ch, sample_buffer_t *func)                                                            
{
  return S_OK;
}





STDMETHODIMP 
DecoderControl::GetPages(CAUUID *pPages)
{
  pPages->cElems = 2;
  pPages->pElems = (GUID *) CoTaskMemAlloc(sizeof(GUID) * pPages->cElems);
  if (pPages->pElems == NULL)
  {
    return E_OUTOFMEMORY;
  }
  (pPages->pElems)[0] = CLSID_AC3Filter_main;
  (pPages->pElems)[1] = CLSID_AC3Filter_mixer;
  return NOERROR;
}


