#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <commctrl.h>
#include <math.h>
#include "resource.h"
#include "dlg_mixer.h"
#include "vac3dec\defs.h"

const double min_gain_level  = -20.0;
const double max_gain_level  = +20.0;
const double min_level = -50.0;
const int ticks = 5;


const int matrix_control[6][6] =
{
  { IDC_EDT_L_L,   IDC_EDT_C_L,   IDC_EDT_R_L,   IDC_EDT_SL_L,   IDC_EDT_SR_L,   IDC_EDT_LFE_L },
  { IDC_EDT_L_C,   IDC_EDT_C_C,   IDC_EDT_R_C,   IDC_EDT_SL_C,   IDC_EDT_SR_C,   IDC_EDT_LFE_C },
  { IDC_EDT_L_R,   IDC_EDT_C_R,   IDC_EDT_R_R,   IDC_EDT_SL_R,   IDC_EDT_SR_R,   IDC_EDT_LFE_R },
  { IDC_EDT_L_SL,  IDC_EDT_C_SL,  IDC_EDT_R_SL,  IDC_EDT_SL_SL,  IDC_EDT_SR_SL,  IDC_EDT_LFE_SL },
  { IDC_EDT_L_SR,  IDC_EDT_C_SR,  IDC_EDT_R_SR,  IDC_EDT_SL_SR,  IDC_EDT_SR_SR,  IDC_EDT_LFE_SR },
  { IDC_EDT_L_LFE, IDC_EDT_C_LFE, IDC_EDT_R_LFE, IDC_EDT_SL_LFE, IDC_EDT_SR_LFE, IDC_EDT_LFE_LFE }
};

const int acmod2control[8] = 
{ IDC_RADIO_2_0, IDC_RADIO_1_0, IDC_RADIO_2_0, IDC_RADIO_3_0, IDC_RADIO_2_1, IDC_RADIO_3_1, IDC_RADIO_2_2, IDC_RADIO_3_2 };
  
#define dlg_printf(dlg, ctrl, format, params)                     \
{                                                                 \
  char buf[255];                                                  \
  sprintf(buf, format, ##params);                                 \
  SendDlgItemMessage(dlg, ctrl, WM_SETTEXT, 0, (LONG)(LPSTR)buf); \
}

#define value2db(value) ((value > 0)? log10(value)*20.0: 0)
#define db2value(db)    pow(10.0, db/20.0)




///////////////////////////////////////////////////////////////////////////////
// Initialization / Deinitialization
///////////////////////////////////////////////////////////////////////////////

CUnknown * WINAPI CAC3Filter_mixer::CreateInstance(LPUNKNOWN lpunk, HRESULT *phr)
{
  DbgLog((LOG_TRACE, 3, "CreateInstance of AC3Filter Mixer property page"));
  CUnknown *punk = new CAC3Filter_mixer(lpunk, phr);
  if (punk == NULL) 
    *phr = E_OUTOFMEMORY;
  return punk;
}


CAC3Filter_mixer::CAC3Filter_mixer(LPUNKNOWN pUnk, HRESULT *phr) 
:CBasePropertyPage(NAME("AC3Filter Mixer Property Page"),pUnk, IDD_MIXER, IDS_MIXER)
{
  DbgLog((LOG_TRACE, 3, "CAC3Filter_mixer::CAC3Filter_mixer()"));
  filter = 0;
  speakers = 0;
  InitCommonControls();
}

HRESULT 
CAC3Filter_mixer::OnConnect(IUnknown *pUnknown)
{
  DbgLog((LOG_TRACE, 3, "CAC3Filter_mixer::OnConnect()"));
  HRESULT hr = pUnknown->QueryInterface(IID_IAC3Filter, (void **) &filter);
  if (FAILED(hr)) 
    return E_NOINTERFACE; 

  ASSERT(filter);
  filter->get_speakers(&speakers);

  return NOERROR;
}


HRESULT 
CAC3Filter_mixer::OnDisconnect()
{
  DbgLog((LOG_TRACE, 3, "CAC3Filter_mixer::OnDisconnect()"));
  if (filter == NULL)
    return E_UNEXPECTED;

  filter->Release();
  filter = 0;
  return NOERROR;
}


HRESULT 
CAC3Filter_mixer::OnActivate()
{
  DbgLog((LOG_TRACE, 3, "CAC3Filter_mixer::OnActivate()"));

  init_controls();
  set_dynamic_controls();
  set_controls();

  SetTimer(m_hwnd, 1, 200, 0);
  return NOERROR;
}

///////////////////////////////////////////////////////////////////////////////
// Handle messages
///////////////////////////////////////////////////////////////////////////////

BOOL 
CAC3Filter_mixer::OnReceiveMessage(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  switch (uMsg)
  {
    case WM_ACTIVATE:
      set_dynamic_controls();
      set_controls();
      return 1;

    case WM_COMMAND:
      command(LOWORD(wParam), HIWORD(wParam));
      return 1;

    case WM_VSCROLL:
      command(GetDlgCtrlID((HWND)lParam), LOWORD(wParam));
      return 1;

    case WM_TIMER:
      set_dynamic_controls();
      return 1;
  }

  return CBasePropertyPage::OnReceiveMessage(hwnd, uMsg, wParam, lParam);
}

///////////////////////////////////////////////////////////////////////////////
// Controls initalization/update
///////////////////////////////////////////////////////////////////////////////

void 
CAC3Filter_mixer::init_controls()
{
  /////////////////////////////////////
  // Link controls

  cmb_matrix.link(m_Dlg, IDC_CMB_MATRIX);
  for (int i = 0; i < 6; i++)
    for (int j = 0; j < 6; j++)
      edt_matrix[i][j].link(m_Dlg, matrix_control[i][j]);

  edt_master.link(m_Dlg, IDC_EDT_MASTER);
  edt_gain  .link(m_Dlg, IDC_EDT_GAIN);
  edt_voice .link(m_Dlg, IDC_EDT_VOICE);
  edt_sur   .link(m_Dlg, IDC_EDT_SUR);
  edt_lfe   .link(m_Dlg, IDC_EDT_LFE);

  edt_gain.enable(false);

  /////////////////////////////////////
  // Init sliders

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_MASTER, TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_MASTER, TBM_SETTIC, 0, 0);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_GAIN,   TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_GAIN,   TBM_SETTIC, 0, 0);

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE,    TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE,    TBM_SETTIC, 0, 0);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE,  TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE,  TBM_SETTIC, 0, 0);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR,    TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR,    TBM_SETTIC, 0, 0);

  /////////////////////////////////////
  // Mixer levels

  int in_ch2control[6]  = { IDC_IN_L,  IDC_IN_C,  IDC_IN_R,  IDC_IN_SL,  IDC_IN_SR,  IDC_IN_LFE  };
  int out_ch2control[6] = { IDC_OUT_L, IDC_OUT_C, IDC_OUT_R, IDC_OUT_SL, IDC_OUT_SR, IDC_OUT_LFE };
  for (int ch = 0; ch < 6; ch++)
  {
    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETBARCOLOR, 0, RGB(0, 128, 0));
    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETBARCOLOR, 0, RGB(0, 128, 0));
    // log scale
    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETRANGE, 0, MAKELPARAM(0, -min_level * ticks));
    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETRANGE, 0, MAKELPARAM(0, -min_level * ticks));
//    // linear scale
//    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETRANGE, 0, MAKELPARAM(0, 256));
//    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETRANGE, 0, MAKELPARAM(0, 256));
  }

}

void 
CAC3Filter_mixer::set_dynamic_controls()
{
  sample_t in_levels[6];
  sample_t out_levels[6];
  sample_t in_level, out_level;
  sample_t master, gain;
  sample_t slev, clev, lfelev;
  bool     slev_lock, clev_lock, lfelev_lock;
  bool     auto_matrix;

  filter->get_levels(in_levels, &in_level, out_levels, &out_level);
  filter->get_gain(&master, &gain);
  filter->get_gains(&slev, &clev, &lfelev);
  filter->get_bsi_locks(&slev_lock, &clev_lock, &lfelev_lock);
  filter->get_auto_matrix(&auto_matrix);

  /////////////////////////////////////
  // Auto gain control

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_GAIN, TBM_SETPOS, TRUE, long(-value2db(gain) * ticks));
  edt_gain.update_value(value2db(gain));

  /////////////////////////////////////
  // Gain controls

  if (clev_lock)   
  { 
    SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE, TBM_SETPOS, TRUE, long(-value2db(clev) * ticks)); 
    EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_VOICE), false);
    edt_voice.update_value(value2db(clev)); 
    edt_voice.enable(false); 
  }

  if (slev_lock)   
  { 
    SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR, TBM_SETPOS, TRUE, long(-value2db(slev) * ticks)); 
    EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_SUR), false);
    edt_sur.update_value(value2db(slev)); 
    edt_sur.enable(false); 
  }

  if (lfelev_lock)   
  { 
    SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE, TBM_SETPOS, TRUE, long(-value2db(lfelev) * ticks)); 
    EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_LFE), false);
    edt_lfe.update_value(value2db(lfelev)); 
    edt_lfe.enable(false); 
  }

  /////////////////////////////////////
  // Mixer levels

  int in_ch2control[6]  = { IDC_IN_L,  IDC_IN_C,  IDC_IN_R,  IDC_IN_SL,  IDC_IN_SR,  IDC_IN_LFE  };
  int out_ch2control[6] = { IDC_OUT_L, IDC_OUT_C, IDC_OUT_R, IDC_OUT_SL, IDC_OUT_SR, IDC_OUT_LFE };
  for (int ch = 0; ch < 6; ch++)
  {
    // log scale
    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETPOS, in_levels[ch] / in_level   > 0? long(-(min_level - log10(in_levels[ch]  / in_level )*20) * ticks): -1000 * ticks,  0);
    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETPOS, out_levels[ch] / out_level > 0? long(-(min_level - log10(out_levels[ch] / out_level)*20) * ticks): -1000 * ticks,  0);
//    // linear scale
//    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETPOS,   long(in_levels[ch] * 256 / in_level),  0);
//    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETPOS,   long(out_levels[ch]* 256 / out_level), 0);
  }

  /////////////////////////////////////
  // Matrix controls

  if (auto_matrix)
    set_matrix_controls();
}

void 
CAC3Filter_mixer::set_controls()
{
  acmod_t  speakers;
  sample_t master, gain;
  sample_t slev, clev, lfelev;
  bool     slev_lock, clev_lock, lfelev_lock;
  bool     auto_matrix;
  bool     normalize_matrix;
  bool     normalize;
  bool     expand_stereo;
  bool     voice_control;

  filter->get_speakers(&speakers);
  filter->get_gain(&master, &gain);
  filter->get_gains(&slev, &clev, &lfelev);
  filter->get_bsi_locks(&slev_lock, &clev_lock, &lfelev_lock);
  filter->get_auto_matrix(&auto_matrix);
  filter->get_normalize_matrix(&normalize_matrix);
  filter->get_normalize(&normalize);
  filter->get_expand_stereo(&expand_stereo);
  filter->get_voice_control(&voice_control);
  
  /////////////////////////////////////
  // Actual output speakers

  for (int acmod = 0; acmod < 8; acmod++)
    SendDlgItemMessage(m_Dlg, acmod2control[acmod], BM_SETCHECK, BST_UNCHECKED, 1);

  SendDlgItemMessage(m_Dlg, acmod2control[MODE_ACMOD(speakers)], BM_SETCHECK, BST_CHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_LFE, BM_SETCHECK, IS_MODE_LFE(speakers)? BST_CHECKED: BST_UNCHECKED, 1);

  /////////////////////////////////////
  // Auto gain control

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_MASTER, TBM_SETPOS, TRUE, long(-value2db(master) * ticks));
  edt_master.update_value(value2db(master));

  /////////////////////////////////////
  // Gain controls

  SendDlgItemMessage(m_Dlg, IDC_CHK_SLOCK,   BM_SETCHECK, slev_lock?   BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_CLOCK,   BM_SETCHECK, clev_lock?   BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_LFELOCK, BM_SETCHECK, lfelev_lock? BST_CHECKED: BST_UNCHECKED, 1);

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE, TBM_SETPOS, TRUE, long(-value2db(clev)   * ticks));
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR,   TBM_SETPOS, TRUE, long(-value2db(slev)   * ticks));
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE,   TBM_SETPOS, TRUE, long(-value2db(lfelev) * ticks));

  EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_VOICE), !clev_lock);
  EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_SUR),   !slev_lock);
  EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_LFE),   !lfelev_lock);

  edt_voice.update_value(value2db(clev));
  edt_sur  .update_value(value2db(slev));
  edt_lfe  .update_value(value2db(lfelev));

  edt_voice.enable(!clev_lock);   
  edt_sur  .enable(!slev_lock);   
  edt_lfe  .enable(!lfelev_lock); 

  /////////////////////////////////////
  // Flags

  SendDlgItemMessage(m_Dlg, IDC_CHK_AUTO_MATRIX,   BM_SETCHECK, auto_matrix?      BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_NORM_MATRIX,   BM_SETCHECK, normalize_matrix? BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_NORM,          BM_SETCHECK, normalize?        BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_EXPAND_STEREO, BM_SETCHECK, expand_stereo?    BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_VOICE_CONTROL, BM_SETCHECK, voice_control?    BST_CHECKED: BST_UNCHECKED, 1);

  /////////////////////////////////////
  // Matrix combo box 

  char matrix_name[256];
  HKEY key;

  SendDlgItemMessage(m_Dlg, IDC_CMB_MATRIX, WM_GETTEXT, 256, (long)matrix_name);
  SendDlgItemMessage(m_Dlg, IDC_CMB_MATRIX, CB_RESETCONTENT, 0, 0);

  if (RegOpenKeyEx(HKEY_CURRENT_USER, "Software\\AC3Filter\\matrix", 0, KEY_READ, &key) == ERROR_SUCCESS)
  {
    char buf[256];
    int i = 0;
    DWORD len = 256;
    while (RegEnumKeyEx(key, i++, (LPTSTR)buf, &len, 0, 0, 0, 0) == ERROR_SUCCESS)
    {
      SendDlgItemMessage(m_Dlg, IDC_CMB_MATRIX, CB_ADDSTRING, 0, (LONG)buf);
      len = 256;
    }
  }
  SendDlgItemMessage(m_Dlg, IDC_CMB_MATRIX, WM_SETTEXT, 0, (long)matrix_name);  
 
  /////////////////////////////////////
  // Matrix

  set_matrix_controls();
}

void 
CAC3Filter_mixer::set_matrix_controls()
{
  bool auto_matrix;
  mixer_matrix_t matrix;
  filter->get_matrix(&matrix);
  filter->get_auto_matrix(&auto_matrix);

  for (int i = 0; i < 6; i++)
    for (int j = 0; j < 6; j++)
    {
      edt_matrix[i][j].update_value(matrix[j][i]);
      SendDlgItemMessage(m_Dlg, matrix_control[j][i], EM_SETREADONLY, auto_matrix, 0);
    }
}

///////////////////////////////////////////////////////////////////////////////
// Commands
///////////////////////////////////////////////////////////////////////////////

void 
CAC3Filter_mixer::command(int control, int message)
{
  /////////////////////////////////////
  // Matrix controls

  if (message == CB_ENTER)
  {
    mixer_matrix_t matrix;
    filter->get_matrix(&matrix);
    bool update_matrix = false;

    for (int i = 0; i < 6; i++)
      for (int j = 0; j < 6; j++)
        if (control == matrix_control[i][j])
        {
          matrix[j][i] = edt_matrix[i][j].value;
          update_matrix = true;
        }

    if (update_matrix)
    {
      filter->set_matrix(&matrix);
      set_matrix_controls();
    }
  }

  switch (control)
  {
    /////////////////////////////////////
    // Speaker selection

    case IDC_RADIO_1_0: speakers = MODE_1_0; goto set_speakers;
    case IDC_RADIO_2_0: speakers = MODE_2_0; goto set_speakers;
    case IDC_RADIO_3_0: speakers = MODE_3_0; goto set_speakers;
    case IDC_RADIO_2_1: speakers = MODE_2_1; goto set_speakers;
    case IDC_RADIO_3_1: speakers = MODE_3_1; goto set_speakers;
    case IDC_RADIO_2_2: speakers = MODE_2_2; goto set_speakers;
    case IDC_RADIO_3_2: speakers = MODE_3_2; goto set_speakers;
    case IDC_CHK_LFE:   speakers = (SendDlgItemMessage(m_Dlg, IDC_CHK_LFE, BM_GETCHECK, 0, 0) == BST_CHECKED)? speakers | MODE_LFE: speakers & ~MODE_LFE;
set_speakers:
    {
      DbgLog((LOG_TRACE, 3, "Speaker radio button presseed"));
      filter->set_speakers(speakers);
      filter->get_speakers(&speakers);
      DbgLog((LOG_TRACE, 3, "Speaker mode set"));
      set_controls();
      DbgLog((LOG_TRACE, 3, "Speaker controls set"));
      break;
    }

    /////////////////////////////////////
    // Auto gain control

    case IDC_SLIDER_MASTER:
      if (message == TB_THUMBPOSITION || message == TB_ENDTRACK)
      {
        sample_t master = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_SLIDER_MASTER,TBM_GETPOS, 0, 0))/ticks);
        filter->set_gain(master);
        set_controls();
      }
      break;

    case IDC_EDT_MASTER:
      if (message == CB_ENTER)
      {
        filter->set_gain(db2value(edt_master.value));
        set_controls();
      }
      break;

    /////////////////////////////////////
    // Gain controls

    case IDC_CHK_CLOCK:    
    case IDC_CHK_SLOCK:    
    case IDC_CHK_LFELOCK:
    {
      bool slev_lock   = (SendDlgItemMessage(m_Dlg, IDC_CHK_SLOCK,   BM_GETCHECK, 0, 0) == BST_CHECKED);
      bool clev_lock   = (SendDlgItemMessage(m_Dlg, IDC_CHK_CLOCK,   BM_GETCHECK, 0, 0) == BST_CHECKED);
      bool lfelev_lock = (SendDlgItemMessage(m_Dlg, IDC_CHK_LFELOCK, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_bsi_locks(slev_lock, clev_lock, lfelev_lock);
      set_controls();
      break;
    }

    case IDC_SLIDER_VOICE:
    case IDC_SLIDER_SUR:
    case IDC_SLIDER_LFE:
      if (message == TB_THUMBPOSITION || message == TB_ENDTRACK)
      {
        sample_t clev   = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE, TBM_GETPOS, 0, 0))/ticks);
        sample_t slev   = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR,   TBM_GETPOS, 0, 0))/ticks);
        sample_t lfelev = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE,   TBM_GETPOS, 0, 0))/ticks);
        filter->set_gains(slev, clev, lfelev);
        set_controls();
      }
      break;

    case IDC_EDT_VOICE:
    case IDC_EDT_SUR:
    case IDC_EDT_LFE:
      if (message == CB_ENTER)
      {
        filter->set_gains(db2value(edt_sur.value), db2value(edt_voice.value), db2value(edt_lfe.value));
        set_controls();
      }
      break;

    /////////////////////////////////////
    // Flags

    case IDC_CHK_AUTO_MATRIX:
    {
      bool auto_matrix = (SendDlgItemMessage(m_Dlg, IDC_CHK_AUTO_MATRIX, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_auto_matrix(auto_matrix);
      set_controls();
      break;
    }

    case IDC_CHK_NORM_MATRIX:
    {
      bool normalize_matrix = (SendDlgItemMessage(m_Dlg, IDC_CHK_NORM_MATRIX, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_normalize_matrix(normalize_matrix);
      set_controls();
      break;
    }

    case IDC_CHK_NORM:
    {
      bool normalize = (SendDlgItemMessage(m_Dlg, IDC_CHK_NORM, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_normalize(normalize);
      set_controls();
      break;
    }

    case IDC_CHK_EXPAND_STEREO:
    {
      bool expand_stereo = (SendDlgItemMessage(m_Dlg, IDC_CHK_EXPAND_STEREO, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_expand_stereo(expand_stereo);
      set_controls();
      break;
    }

    case IDC_CHK_VOICE_CONTROL:
    {
      bool voice_control = (SendDlgItemMessage(m_Dlg, IDC_CHK_VOICE_CONTROL, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_voice_control(voice_control);
      set_controls();
      break;
    }

    /////////////////////////////////////
    // Matrix combo box 

    case IDC_BTN_SAVE:
    {
      char buf[256];
      SendDlgItemMessage(m_Dlg, IDC_CMB_MATRIX, WM_GETTEXT, 256, (LONG)buf);
      filter->save_matrix(buf);
      set_controls();
      break;
    }

    case IDC_BTN_LOAD:
    {
      char buf[256];
      SendDlgItemMessage(m_Dlg, IDC_CMB_MATRIX, WM_GETTEXT, 256, (LONG)buf);
      filter->set_auto_matrix(false);
      filter->load_matrix(buf);
      set_controls();
      break;
    }
  }
}
