#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <commctrl.h>
#include <math.h>
#include "resource.h"
#include "dlg_main.h"
#include "vac3dec\defs.h"

const double min_gain_level  = -20.0;
const double max_gain_level  = +20.0;
const double min_level = -50.0;
const int ticks = 5;


const int acmod2control[8] = 
{ IDC_RADIO_2_0, IDC_RADIO_1_0, IDC_RADIO_2_0, IDC_RADIO_3_0, IDC_RADIO_2_1, IDC_RADIO_3_1, IDC_RADIO_2_2, IDC_RADIO_3_2 };
  
#define dlg_printf(dlg, ctrl, format, params)                     \
{                                                                 \
  char buf[255];                                                  \
  sprintf(buf, format, ##params);                                 \
  SendDlgItemMessage(dlg, ctrl, WM_SETTEXT, 0, (LONG)(LPSTR)buf); \
}

#define value2db(value) ((value > 0)? log10(value)*20.0: 0)
#define db2value(db)    pow(10.0, (db)/20.0)




///////////////////////////////////////////////////////////////////////////////
// Initialization / Deinitialization
///////////////////////////////////////////////////////////////////////////////

CUnknown * WINAPI CAC3Filter_main::CreateInstance(LPUNKNOWN lpunk, HRESULT *phr)
{
  DbgLog((LOG_TRACE, 3, "CreateInstance of AC3Filter Main property page"));
  CUnknown *punk = new CAC3Filter_main(lpunk, phr);
  if (punk == NULL) 
    *phr = E_OUTOFMEMORY;
  return punk;
}

CAC3Filter_main::CAC3Filter_main(LPUNKNOWN pUnk, HRESULT *phr) 
:CBasePropertyPage(NAME("AC3Filter Mixer Property Page"),pUnk, IDD_MAIN, IDS_MAIN)
{
  DbgLog((LOG_TRACE, 3, "CAC3Filter_main::CAC3Filter_main()"));
  filter = 0;
  speakers = 0;
  InitCommonControls();
}

HRESULT 
CAC3Filter_main::OnConnect(IUnknown *pUnknown)
{
  DbgLog((LOG_TRACE, 3, "CAC3Filter_main::OnConnect()"));
  HRESULT hr = pUnknown->QueryInterface(IID_IAC3Filter, (void **) &filter);
  if (FAILED(hr)) 
    return E_NOINTERFACE; 

  ASSERT(filter);
  filter->get_speakers(&speakers);

  return NOERROR;
}

HRESULT 
CAC3Filter_main::OnDisconnect()
{
  DbgLog((LOG_TRACE, 3, "CAC3Filter_main::OnDisconnect()"));
  if (filter == NULL)
    return E_UNEXPECTED;

  filter->Release();
  filter = 0;
  return NOERROR;
}

HRESULT 
CAC3Filter_main::OnActivate()
{
  DbgLog((LOG_TRACE, 3, "CAC3Filter_main::OnActivate()"));

  init_controls();
  set_dynamic_controls();
  set_controls();

  SetTimer(m_hwnd, 1, 200, 0);
  return NOERROR;
}

///////////////////////////////////////////////////////////////////////////////
// Handle messages
///////////////////////////////////////////////////////////////////////////////

BOOL 
CAC3Filter_main::OnReceiveMessage(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  switch (uMsg)
  {
    case WM_ACTIVATE:
      set_dynamic_controls();
      set_controls();
      return 1;

    case WM_COMMAND:
      command(LOWORD(wParam), HIWORD(wParam));
      return 1;

    case WM_VSCROLL:
      command(GetDlgCtrlID((HWND)lParam), LOWORD(wParam));
      return 1;

    case WM_TIMER:
      set_dynamic_controls();
      return 1;
  }

  return CBasePropertyPage::OnReceiveMessage(hwnd, uMsg, wParam, lParam);
}

///////////////////////////////////////////////////////////////////////////////
// Controls initalization/update
///////////////////////////////////////////////////////////////////////////////

void 
CAC3Filter_main::init_controls()
{
  /////////////////////////////////////
  // Init sliders

  SendDlgItemMessage(m_Dlg, IDC_DRC_LEVEL,     TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_DRC_LEVEL,     TBM_SETTIC, 0, 0);
  SendDlgItemMessage(m_Dlg, IDC_DRC_POWER,     TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_DRC_POWER,     TBM_SETTIC, 0, 0);

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_MASTER, TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_MASTER, TBM_SETTIC, 0, 0);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_GAIN,   TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_GAIN,   TBM_SETTIC, 0, 0);

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE,    TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE,    TBM_SETTIC, 0, 0);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE,  TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE,  TBM_SETTIC, 0, 0);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR,    TBM_SETRANGE, TRUE, MAKELONG(min_gain_level, max_gain_level) * ticks);
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR,    TBM_SETTIC, 0, 0);

  SendDlgItemMessage(m_Dlg, IDC_CPU,           PBM_SETRANGE, 0, MAKELPARAM(0, 100));

  /////////////////////////////////////
  // Mixer levels

  int in_ch2control[6]  = { IDC_IN_L,  IDC_IN_C,  IDC_IN_R,  IDC_IN_SL,  IDC_IN_SR,  IDC_IN_LFE  };
  int out_ch2control[6] = { IDC_OUT_L, IDC_OUT_C, IDC_OUT_R, IDC_OUT_SL, IDC_OUT_SR, IDC_OUT_LFE };
  for (int ch = 0; ch < 6; ch++)
  {
    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETBARCOLOR, 0, RGB(0, 128, 0));
    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETBARCOLOR, 0, RGB(0, 128, 0));
    // log scale
    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETRANGE, 0, MAKELPARAM(0, -min_level * ticks));
    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETRANGE, 0, MAKELPARAM(0, -min_level * ticks));
//    // linear scale
//    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETRANGE, 0, MAKELPARAM(0, 256));
//    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETRANGE, 0, MAKELPARAM(0, 256));
  }

}

void 
CAC3Filter_main::set_dynamic_controls()
{
  BSI      bsi;
  int      frames, errors;
  double   cpu_load;
  sample_t in_levels[6];
  sample_t out_levels[6];
  sample_t in_level, out_level;
  bool     dynrng;
  sample_t dynrng_level, dynrng_power;
  sample_t master, gain;
  sample_t slev, clev, lfelev;
  bool     slev_lock, clev_lock, lfelev_lock;

  filter->get_bsi(&bsi);
  filter->get_dynrng(&dynrng, &dynrng_level, &dynrng_power);
  filter->get_levels(in_levels, &in_level, out_levels, &out_level);
  filter->get_gain(&master, &gain);
  filter->get_gains(&slev, &clev, &lfelev);
  filter->get_bsi_locks(&slev_lock, &clev_lock, &lfelev_lock);
  filter->get_cpu_load(&cpu_load);
  filter->get_stat(&frames, &errors);

  /////////////////////////////////////
  // Speaker configuration

  char buffer[255];
  const char *modes[8] = { "1+1 (Dual mono)", "1/0 (Mono)", "2/0 (Stereo)", "3/0", "2/1", "3/1", "2/2 (Quadro)", "3/2" };
  strcpy(buffer, modes[MODE_ACMOD(bsi.acmod)]);
  if (IS_MODE_DOLBY(bsi.acmod))
    strcat(buffer, "Dolby");
  if (IS_MODE_LFE(bsi.acmod))
    strcat(buffer, "+LFE");
  SendDlgItemMessage(m_Dlg, IDC_EDT_CHANNELS, WM_SETTEXT, 0, (DWORD) (LPSTR) buffer);

  /////////////////////////////////////
  // Stream information

  dlg_printf(m_Dlg, IDC_EDT_BITRATE,       "%d", bsi.bitrate);
  dlg_printf(m_Dlg, IDC_EDT_SAMPLE_RATE,   "%d", bsi.sample_rate);
  dlg_printf(m_Dlg, IDC_EDT_FREQ_COUPLING, "%2.3f", double(bsi.cplbegf)/1000);
  dlg_printf(m_Dlg, IDC_EDT_FREQ_HIGH,     "%2.3f", double(bsi.chendf[0])/1000);
  dlg_printf(m_Dlg, IDC_EDT_FRAMES,        "%i", frames);
  dlg_printf(m_Dlg, IDC_EDT_ERRORS,        "%i", errors);
  dlg_printf(m_Dlg, IDC_CPU_LABEL,         "%i%%", int(cpu_load*100));
  SendDlgItemMessage(m_Dlg, IDC_CPU, PBM_SETPOS, int(cpu_load * 100),  0);

  /////////////////////////////////////
  // Dynamic range compression

  SendDlgItemMessage(m_Dlg, IDC_DRC_LEVEL, TBM_SETPOS, TRUE, long(-value2db(dynrng_level) * ticks));

  /////////////////////////////////////
  // Auto gain control

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_GAIN, TBM_SETPOS, TRUE, long(-value2db(gain) * ticks));

  /////////////////////////////////////
  // Gain controls

  if (clev_lock)   
  { 
    SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE, TBM_SETPOS, TRUE, long(-value2db(clev) * ticks)); 
    EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_VOICE), false);
  }

  if (slev_lock)   
  { 
    SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR, TBM_SETPOS, TRUE, long(-value2db(slev) * ticks)); 
    EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_SUR), false);
  }

  if (lfelev_lock)   
  { 
    SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE, TBM_SETPOS, TRUE, long(-value2db(lfelev) * ticks)); 
    EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_LFE), false);
  }

  /////////////////////////////////////
  // Mixer levels

  int in_ch2control[6]  = { IDC_IN_L,  IDC_IN_C,  IDC_IN_R,  IDC_IN_SL,  IDC_IN_SR,  IDC_IN_LFE  };
  int out_ch2control[6] = { IDC_OUT_L, IDC_OUT_C, IDC_OUT_R, IDC_OUT_SL, IDC_OUT_SR, IDC_OUT_LFE };
  for (int ch = 0; ch < 6; ch++)
  {
    // log scale
    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETPOS, in_levels[ch] / in_level   > 0? long(-(min_level - log10(in_levels[ch]  / in_level )*20) * ticks): -1000 * ticks,  0);
    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETPOS, out_levels[ch] / out_level > 0? long(-(min_level - log10(out_levels[ch] / out_level)*20) * ticks): -1000 * ticks,  0);
//    // linear scale
//    SendDlgItemMessage(m_Dlg, in_ch2control[ch],  PBM_SETPOS,   long(in_levels[ch] * 256 / in_level),  0);
//    SendDlgItemMessage(m_Dlg, out_ch2control[ch], PBM_SETPOS,   long(out_levels[ch]* 256 / out_level), 0);
  }
}

void 
CAC3Filter_main::set_controls()
{
  acmod_t  speakers;
  sample_t master, gain;
  sample_t slev, clev, lfelev;
  bool     slev_lock, clev_lock, lfelev_lock;
  bool     normalize;
  bool     dynrng;
  sample_t dynrng_level, dynrng_power;

  filter->get_speakers(&speakers);
  filter->get_gain(&master, &gain);
  filter->get_gains(&slev, &clev, &lfelev);
  filter->get_bsi_locks(&slev_lock, &clev_lock, &lfelev_lock);
  filter->get_normalize(&normalize);
  filter->get_dynrng(&dynrng, &dynrng_level, &dynrng_power);
  
  /////////////////////////////////////
  // Actual output speakers

  for (int acmod = 0; acmod < 8; acmod++)
    SendDlgItemMessage(m_Dlg, acmod2control[acmod], BM_SETCHECK, BST_UNCHECKED, 1);

  SendDlgItemMessage(m_Dlg, acmod2control[MODE_ACMOD(speakers)], BM_SETCHECK, BST_CHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_LFE, BM_SETCHECK, IS_MODE_LFE(speakers)? BST_CHECKED: BST_UNCHECKED, 1);

  /////////////////////////////////////
  // Dynamic range compression

  SendDlgItemMessage(m_Dlg, IDC_CHK_DYNRNG, BM_SETCHECK, dynrng? BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_DRC_POWER, TBM_SETPOS, TRUE, long(-value2db(dynrng_power) * ticks));
                                                                                          
  /////////////////////////////////////
  // Auto gain control

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_MASTER, TBM_SETPOS, TRUE, long(-value2db(master) * ticks));

  /////////////////////////////////////
  // Gain controls

  SendDlgItemMessage(m_Dlg, IDC_CHK_SLOCK,   BM_SETCHECK, slev_lock?   BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_CLOCK,   BM_SETCHECK, clev_lock?   BST_CHECKED: BST_UNCHECKED, 1);
  SendDlgItemMessage(m_Dlg, IDC_CHK_LFELOCK, BM_SETCHECK, lfelev_lock? BST_CHECKED: BST_UNCHECKED, 1);

  SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE, TBM_SETPOS, TRUE, long(-value2db(clev)   * ticks));
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR,   TBM_SETPOS, TRUE, long(-value2db(slev)   * ticks));
  SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE,   TBM_SETPOS, TRUE, long(-value2db(lfelev) * ticks));

  EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_VOICE), !clev_lock);
  EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_SUR),   !slev_lock);
  EnableWindow(GetDlgItem(m_Dlg, IDC_SLIDER_LFE),   !lfelev_lock);

  /////////////////////////////////////
  // Flags

  SendDlgItemMessage(m_Dlg, IDC_CHK_NORMALIZE, BM_SETCHECK, normalize? BST_CHECKED: BST_UNCHECKED, 1);
}


///////////////////////////////////////////////////////////////////////////////
// Commands
///////////////////////////////////////////////////////////////////////////////

void 
CAC3Filter_main::command(int control, int message)
{
  switch (control)
  {
    /////////////////////////////////////
    // Speaker selection

    case IDC_RADIO_1_0: speakers = MODE_1_0; goto set_speakers;
    case IDC_RADIO_2_0: speakers = MODE_2_0; goto set_speakers;
    case IDC_RADIO_3_0: speakers = MODE_3_0; goto set_speakers;
    case IDC_RADIO_2_1: speakers = MODE_2_1; goto set_speakers;
    case IDC_RADIO_3_1: speakers = MODE_3_1; goto set_speakers;
    case IDC_RADIO_2_2: speakers = MODE_2_2; goto set_speakers;
    case IDC_RADIO_3_2: speakers = MODE_3_2; goto set_speakers;
    case IDC_CHK_LFE:   speakers = (SendDlgItemMessage(m_Dlg, IDC_CHK_LFE, BM_GETCHECK, 0, 0) == BST_CHECKED)? speakers | MODE_LFE: speakers & ~MODE_LFE;
set_speakers:
    {
      DbgLog((LOG_TRACE, 3, "Speaker radio button presseed"));
      filter->set_speakers(speakers);
      filter->get_speakers(&speakers);
      DbgLog((LOG_TRACE, 3, "Speaker mode set"));
      set_controls();
      DbgLog((LOG_TRACE, 3, "Speaker controls set"));
      break;
    }

    /////////////////////////////////////
    // Auto gain control

    case IDC_SLIDER_MASTER:
      if (message == TB_THUMBPOSITION || message == TB_ENDTRACK)
      {
        sample_t master = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_SLIDER_MASTER,TBM_GETPOS, 0, 0))/ticks);
        filter->set_gain(master);
        set_controls();
      }
      break;

    /////////////////////////////////////
    // Gain controls

    case IDC_CHK_CLOCK:    
    case IDC_CHK_SLOCK:    
    case IDC_CHK_LFELOCK:
    {
      bool slev_lock   = (SendDlgItemMessage(m_Dlg, IDC_CHK_SLOCK,   BM_GETCHECK, 0, 0) == BST_CHECKED);
      bool clev_lock   = (SendDlgItemMessage(m_Dlg, IDC_CHK_CLOCK,   BM_GETCHECK, 0, 0) == BST_CHECKED);
      bool lfelev_lock = (SendDlgItemMessage(m_Dlg, IDC_CHK_LFELOCK, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_bsi_locks(slev_lock, clev_lock, lfelev_lock);
      set_controls();
      break;
    }

    case IDC_SLIDER_VOICE:
    case IDC_SLIDER_SUR:
    case IDC_SLIDER_LFE:
      if (message == TB_THUMBPOSITION || message == TB_ENDTRACK)
      {
        sample_t clev   = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_SLIDER_VOICE, TBM_GETPOS, 0, 0))/ticks);
        sample_t slev   = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_SLIDER_SUR,   TBM_GETPOS, 0, 0))/ticks);
        sample_t lfelev = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_SLIDER_LFE,   TBM_GETPOS, 0, 0))/ticks);
        filter->set_gains(slev,  clev,  lfelev);
        set_controls();
      }
      break;

    /////////////////////////////////////
    // Dynamic range compreesion

    case IDC_DRC_POWER:
      if (message == TB_THUMBPOSITION || message == TB_ENDTRACK)
      {
        bool dynrng;
        sample_t dynrng_level, dynrng_power;
        filter->get_dynrng(&dynrng, &dynrng_level, &dynrng_power);
        dynrng_power = db2value(-double(SendDlgItemMessage(m_Dlg, IDC_DRC_POWER, TBM_GETPOS, 0, 0))/ticks);
        filter->set_dynrng(dynrng, dynrng_power);
        set_controls();
      }
      break;

    case IDC_CHK_DYNRNG:
    {
      bool dynrng;
      sample_t dynrng_level, dynrng_power;
      filter->get_dynrng(&dynrng, &dynrng_level, &dynrng_power);
      dynrng = (SendDlgItemMessage(m_Dlg, IDC_CHK_DYNRNG, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_dynrng(dynrng, dynrng_power);
      set_controls();
      break;
    }


    /////////////////////////////////////
    // Flags

    case IDC_CHK_NORMALIZE:
    {
      bool normalize = (SendDlgItemMessage(m_Dlg, IDC_CHK_NORMALIZE, BM_GETCHECK, 0, 0) == BST_CHECKED);
      filter->set_normalize(normalize);
      set_controls();
      break;
    }
  }
}

